(ns zing.zip
  "A zipper library with small alterations from clojure.zip.
  Uses records and a protocol for zippers rather than a 2 element vector and
  metadata protocol implementation.
  A regular map with :node and :path keys can be used, the zipper fns will be
  looked up at ::branch?/::children/::make-node keys
  Unlike the clojure.zip impl, the make-node fn always uses 3 args
  Uses nil when navigating past the end rather than the sentinel value
  that clojure.zip uses, which is how navigating up past the top or left or
  right past the children works.
  Has more implementations than just seqs, vectors and clojure.xml maps,
  also has alternate map trees with :label and :children keys, edn collections
  with mixed lists, vectors, sets and maps, and hiccup.
  Note: the implementations of the branch/children/make-node functions are
  different than usual, they accept a loc rather than a node" 
  (:refer-clojure :exclude [find replace remove next])
  #?(:clj (:import [clojure.lang IDeref IPersistentMap MapEntry])
     :cljs (:require [cljs.core :refer [IDeref IMap MapEntry]])))

(defrecord ZipperPath [l pnodes ppath r changed?])

(defn zipper-path
  "ZipperPath constructor, changed? is optional, defaults to false"
  ([l pnodes ppath r] (zipper-path l pnodes ppath r false))
  ([l pnodes ppath r changed?]
   (ZipperPath. l pnodes ppath r changed?)))

(defprotocol Zipper
  (-branch? [loc])
  (-children [loc])
  (-make-node [loc node children]))

#?(:clj
   (extend-type IPersistentMap
     Zipper
     (-branch? [loc] ((::branch? loc) loc))
     (-children [loc] ((::children loc) loc))
     (-make-node [this loc children] ((::make-node this) this loc children))))

#?(:cljs
   (extend-type PersistentArrayMap
       Zipper
       (-branch? [loc] ((::branch? loc) loc))
       (-children [loc] ((::children loc) loc))
       (-make-node [this loc children] ((::make-node this) this loc children))))

#?(:cljs
   (extend-type PersistentHashMap
       Zipper
       (-branch? [loc] ((::branch? loc) loc))
       (-children [loc] ((::children loc) loc))
       (-make-node [this loc children] ((::make-node this) this loc children))))

#?(:cljs
   (extend-type PersistentTreeMap
     Zipper
     (-branch? [loc] ((::branch? loc) loc))
     (-children [loc] ((::children loc) loc))
     (-make-node [this loc children] ((::make-node this) this loc children))))

(defrecord SeqZipper [node path]
  Zipper
  (-branch? [loc] (seq? (.-node ^SeqZipper loc)))
  (-children [loc] (seq (.-node ^SeqZipper loc)))
  (-make-node [this loc children] children))

(defn seq-zip
  "A zipper for nested sequences"
  [root]
  (SeqZipper. root nil))

(defrecord VectorZipper [node path]
  Zipper
  (-branch? [loc] (vector? (.-node ^VectorZipper loc)))
  (-children [loc] (seq (.-node ^VectorZipper loc)))
  (-make-node [loc node children] (vec children)))

(defn vector-zip
  "A zipper for nested vectors"
  [root]
  (VectorZipper. root nil))

(defrecord XMLZipper [node path]
  Zipper
  (-branch? [loc] (not (string? (.-node ^XMLZipper loc))))
  (-children [loc] (seq (:content (.-node ^XMLZipper loc))))
  (-make-node [loc node children]
    (assoc node :content (and children (apply vector children)))))

(defn xml-zip
  "A zipper for xml maps from xml/parse, like the clojure.zip function"
  [root]
  (XMLZipper. root nil))

(defrecord TreeZipper [node path]
  Zipper
  (-branch? [loc] (let [node (.-node ^TreeZipper loc)] (and (map? node) (:label node))))
  (-children [loc] (seq (:children (.-node ^TreeZipper loc))))
  (-make-node [loc node children]
    (assoc node :children (and children (apply vector children)))))

(defn tree-zip
  "A zipper for maps with branches having a :label key and children under the :children key"
  [root]
  (TreeZipper. root nil))

(defn edn-make-node
  "Creates a branch node for an EDNZipper given an existing node and a seq of children"
  [node children]
  (cond
    (vector? node) (vec children)
    (set? node) (set children)
    (map? node)
    (persistent!
     (reduce (fn [acc [k v]] (assoc! acc k v))
             (transient {})
             children))
    (map-entry? node) (let [[k v] children] #?(:clj (MapEntry. k v) :cljs (MapEntry. k v nil)))
    :else (seq children)))

(defrecord EDNZipper [node path]
  Zipper
  (-branch? [loc] (coll? (.-node ^EDNZipper loc)))
  (-children [loc] (seq (.-node ^EDNZipper loc)))
  (-make-node [loc node children] (edn-make-node node children)))

(defn edn-zip
  "A zipper for a nested collection of vectors, sets and maps"
  [root]
  (EDNZipper. root nil))

(defn hiccup-children
  "Given a branch node for a hiccup zipper, returns a seq of the children"
  [node]
  (if (vector? node)
    (if (map? (second node))
      (drop 2 node)
      (drop 1 node))
    node))

(defn hiccup-make-node
  "Make a branch node for a hiccup zipper given a node and children"
  [node children]
  (if (vector? node)
    (let [maybe-attr (second node)]
      (if (map? maybe-attr)
        (into [(first node) maybe-attr] children)
        (into [(first node)] children)))
    children))

(defrecord HiccupZipper [node path]
  Zipper
  (-branch? [loc] (sequential? (.-node ^HiccupZipper loc)))
  (-children [loc] (hiccup-children (.-node ^HiccupZipper loc)))
  (-make-node [loc node children] (hiccup-make-node node children)))

(defn hiccup-zip
  "A zipper for a hiccup tree"
  [root]
  (HiccupZipper. root nil))

(defn node
  "Returns the node at loc"
  [loc]
  (:node loc))

(defn path
  "Returns a seq of nodes leading to this loc"
  [loc]
  (.-pnodes ^ZipperPath (:path loc)))

(defn branch?
  "Returns true if the node at loc is a branch"
  [loc]
  (-branch? loc))

(defn children
  "Returns a seq of the children at loc, which must be a branch"
  [loc]
  (when (branch? loc)
    (-children loc)))

(defn make-node
  "Returns a new branch node, give a loc, an existing node and children. Loc is
  used only for protocol dispatch"
  [loc node children]
  (-make-node loc node children))

(defn lefts
  "Returns a seq of the left siblings of the loc"
  [loc]
  (seq (.-l ^ZipperPath (:path loc))))

(defn rights
  "Returns a seq of the right siblings of the loc"
  [loc]
  (seq (.-r ^ZipperPath (:path loc))))

(defn right
  "Returns the loc of the right sibling of the node at this loc, or nil"
  [loc]
  (let [{node :node path :path} loc
        {l :l [fr & nr :as r] :r} path]
    (when (and path r)
      (assoc loc
             :node fr
             :path (assoc path
                          :l (conj l node)
                          :r nr)))))

(defn rightmost
  "Returns the loc of the rightmost sibling of this loc, or nil"
  [loc]
  (let [{node :node {l :l r :r :as path} :path} loc]
    (if (and path r)
      (assoc loc
             :node (last r)
             :path (assoc path :l (apply conj l node (butlast r)) :r nil))
      loc)))

(defn left
  "Returns the loc of the left sibling of the node at this loc, or nil"
  [loc]
  (let [{node :node {l :l r :r :as path} :path} loc]
    (when (and path (seq l))
      (assoc loc
             :node (peek l)
             :path (assoc path
                          :l (pop l)
                          :r (cons node r))))))

(defn leftmost
  "Returns the loc of the leftmost sibling of this loc, or nil"
  [loc]
  (let [{node :node {l :l r :r :as path} :path} loc]
    (if (and path (seq l))
      (assoc loc
             :node (first l)
             :path (assoc path :l [] :r (concat (rest l) [node] r)))
      loc)))

(defn down
  "Returns the loc of the leftmost child of the node at loc or nil if no children"
  [loc]
  (when (branch? loc)
    (let [node (:node loc)
          path (:path loc)
          [c & cs :as children] (children loc)]
      (when children
        (assoc loc
               :node c
               :path (zipper-path []
                                  (if path (conj (:pnodes path) node) [node])
                                  path
                                  cs))))))

(defn up
  "Returns the loc of the parent of the node at loc or nil if at the top"
  [loc]
  (let [node (:node loc)
        {l :l ppath :ppath pnodes :pnodes r :r changed? :changed?} (:path loc)]
    (when pnodes
      (let [pnode (peek pnodes)]
        (if changed?
          (assoc loc
                 :node (make-node loc pnode (concat l (cons node r)))
                 :path (and ppath (assoc ppath :changed? true)))
          (assoc loc :node pnode :path ppath))))))

(defn root-loc
  "Zips all the way up to the root loc, reflecting changes"
  [loc]
  (when loc
    (let [p (up loc)]
      (if p
        (recur p)
        loc))))

(defn root
  "Zips all the way up and returns the root node, reflecting changes"
  [loc]
  (node (root-loc loc)))

(defn next
  "Moves to the next loc in a depth-first traversal, or nil if at the end"
  [loc]
  (when loc
    (or
     (and (branch? loc) (down loc))
     (right loc)
     (loop [p loc]
       (when-let [parent-p (up p)] 
         (or (right parent-p)
             (recur (up p))))))))

(defn next-leaf
  "Moves to the next leaf node loc in a depth-first traversal, or nil if at the end"
  [loc]
  (let [loc (next loc)]
    (when loc
      (if (branch? loc)
        (recur loc)
        loc))))

(defn prev
  "Moves to the previous node loc in depth-first order, or nil if at the top"
  [loc]
  (if-let [lloc (left loc)]
    (loop [loc lloc]
      (if-let [child (and (branch? loc) (down loc))]
        (recur (rightmost child))
        loc))
    (up loc)))

(defn children-locs
  "Returns a seq of the locs of the children of loc, an empty list if there are
  no children"
  [loc]
  (if (branch? loc)
    (->> (iterate right (down loc)) (take-while some?))
    (list)))

(defn loc-pred
  "Given a node predicate fn, returns a loc predicate fn"
  [node-pred]
  (comp node-pred node))

(defn find
  "Moves forward, in depth-first order, until the loc predicate matches, or nil
  if there are no matches"
  [loc pred]
  (when loc
    (if (pred loc)
      loc
      (recur (next loc) pred))))

(defn find-reverse
  "Moves backward, in depth-first order, until the loc predicate matches, or nil
  if there are no matches"
  [loc pred]
  (when loc
    (if (pred loc)
      loc
      (recur (prev loc) pred))))

(defn search
  "Returns a vector of locs that match the predicate when searching forward,
  in depth-first order, from the loc"
  ([loc pred] (search loc pred []))
  ([loc pred results]
   (if loc
     (if (pred loc)
       (recur (next loc) pred (conj results loc))
       (recur (next loc) pred results))
     results)))

(defn search-reverse
  "Returns a vector of locs that match the predicate when searching backward,
  in depth-first order, from the loc"
  ([loc pred] (search-reverse loc pred []))
  ([loc pred results]
   (if loc
     (if (pred loc)
       (recur (prev loc) pred (conj results loc))
       (recur (prev loc) pred results))
     results)))

(defn insert-left
  "Inserts the item as the left sibling of the node at this loc, without moving"
  [loc item]
  (let [{node :node {l :l :as path} :path} loc]
    (if path
      (assoc loc
             :node node
             :path (assoc path :l (conj l item) :changed? true))
      (throw (ex-info "Insert at top" {:loc loc :item item :root (root loc)})))))

(defn insert-right
  "Inserts the item as the right sibling of the node at this loc, without moving"
  [loc item]
  (let [{node :node {r :r :as path} :path} loc]
    (if path
      (assoc loc
             :node node
             :path (assoc path :r (cons item r) :changed? true))
      (throw (ex-info "Insert at top" {:loc loc :item item :root (root loc)})))))

(defn replace
  "Replaces the node at this loc, without moving"
  [loc node]
  (let [path (:path loc)]
    (assoc loc
           :node node
           :path (assoc path :changed? true))))

(defn edit
  "Replaces the node at this loc with the value of (f node args)"
  [loc f & args]
  (replace loc (apply f (node loc) args)))

(defn insert-child
  "Inserts the item as the leftmost child of the node at this loc,
  without moving"
  [loc item]
  (replace loc (make-node loc (node loc) (cons item (children loc)))))

(defn append-child
  "Inserts the item as the rightmost child of the node at this loc,
  without moving"
  [loc item]
  (replace loc (make-node loc (node loc) (concat (children loc) [item]))))

(defn remove
  "Removes the node at loc, returning the loc that would have preceded it in
  depth-first order"
  [loc]
  (let [{node :node {l :l, ppath :ppath, pnodes :pnodes, r :r :as path} :path} loc]
    (if (nil? path)
      (throw (ex-info "Remove at top" {:loc loc, :root (root loc)}))
      (if (pos? (count l))
        (loop [loc (assoc loc
                          :node (peek l)
                          :path (assoc path :l (pop l) :changed? true))]
          (if-let [child (and (branch? loc) (down loc))]
            (recur (rightmost child))
            loc))
        (assoc loc
               :node (make-node loc (peek pnodes) r)
               :path (and ppath (assoc ppath :changed? true)))))))

(defprotocol Labeled
  (-label [mark]))

(deftype Mark [val label]
  IDeref
  #?(:clj (deref [_] val)
     :cljs (-deref [_] val))
  Labeled
  (-label [_] label))

(defn label
  "Returns the label for this mark"
  [mark]
  (-label mark))

(defn mark
  "Creates a mark, with an optional label (defaults to zing.zip/mark)"
  ([x] (mark x ::mark))
  ([x label]
   (Mark. x label)))

(defn mark-loc
  "Mark this loc, with an optional label"
  ([loc] (mark-loc loc ::mark))
  ([loc label]
   (edit loc mark label)))

(defn marked?
  "Returns true if the node at loc is a Mark"
  [loc]
  (= Mark (type (node loc))))

(defn rebound
  "Moves forward, in depth-first order, until pred matches, calls f on the loc
  that matched, then moves back from the returned value to the original loc.
  f must return a loc."
  [loc pred f]
  (let [loc (mark-loc loc)
        found (find loc pred)
        updated (f found)]
    (edit (find-reverse updated marked?) deref)))
