(ns betfairsports.network
  (:require [clj-http.client :as client]
                [cheshire.core :as cheshire]
                [clojure.set]))

(def ^:dynamic *cm* (clj-http.conn-mgr/make-reusable-conn-manager {:timeout 10 :threads 5 :insecure? true}))
(def ^{:dynamic true} *options* {})

(def ^{:dynamic true} *authentication* "")
(def ^{:dynamic true} *application* "")

(defn endpoint-map []
  {
     :api7-betting "https://api.betfair.com/exchange/betting/json-rpc/v1"
     :api7-account "https://api.betfair.com/exchange/account/json-rpc/v1"
     :api7-scores "https://api.betfair.com/exchange/scores/json-rpc/v1"
  })

(defn- endpoints [endpoint-key] 
  ((endpoint-map) endpoint-key))

(defn endpoint-key-from-value [full-address]
  ((clojure.set/map-invert (endpoint-map)) full-address))

(defn- uuid [] (str (java.util.UUID/randomUUID)))

(defn get-auth-headers []
  (let [headers {"X-Authentication" *authentication* "X-Application" *application*}]
    (into {} (filter (fn [v] (not (clojure.string/blank? (str (v 1))))) headers))))

(defn- get-json-rpc-call-body [method params id]
  (cheshire/generate-string {"jsonrpc" "2.0" "method" method "params" params "id" id}))

(defn do-get 
  "Posts the body provided to the resource at the endpoint symbol provided (looked up in endpoint-map)
  with the headers."
  ([url] (do-get url {} {} :json))
  ([url params headers] (do-get url params headers :json))
  ([url params headers req-accept] 
    {:pre  [(not (nil? url))]}
    (do 
        (let [options  {:headers headers
                       :query-params params
                       :content-type req-accept
                       :socket-timeout 2000  ;; in milliseconds
                       :conn-timeout 2000   ;; in milliseconds
                       :accept req-accept
                       :connection-manager *cm*
                       :as :json
                       :save-request? true}]
        (client/with-middleware (filter #(not (= client/wrap-lower-case-headers %)) client/default-middleware)
          (client/get (str url) (conj options *options*))) ))))

(defn do-post 
  "Posts the body provided to the resource at the url provided with the headers."
  ([url body] (do-post url body {} :json))
  ([url body headers] (do-post url body headers :json))
  ([url body headers req-accept] 
    {:pre  [(not (nil? url))]}
    (do 
        (let [options  {:body body
                       :headers headers
                       :content-type req-accept
                       :socket-timeout 2000  ;; in milliseconds
                       :conn-timeout 2000   ;; in milliseconds
                       :accept req-accept
                       :connection-manager *cm*
                       :as :json
                       :save-request? true}]
        (client/with-middleware (filter #(not (= client/wrap-lower-case-headers %)) client/default-middleware)
          (client/post (str url) (conj options *options*))) ))))

(defn do-post-endpoint
  "Posts the body provided to the resource at the endpoint symbol provided (looked up in endpoint-map)
  with the headers."
  ([endpoint body] (do-post-endpoint endpoint body {} :json))
  ([endpoint body headers] (do-post-endpoint endpoint body headers :json))
  ([endpoint body headers req-accept] 
    (do-post (endpoints endpoint) body headers req-accept)))

(defn do-json-rpc 
  "Performs a json-rpc request using the do-post method against the endpoint with the method, params."
  [endpoint method params] 
    (let [headers (get-auth-headers)
          params-without-nil (into {} (filter second params))
          json-rpc-body (get-json-rpc-call-body method params-without-nil (uuid))]
          (do-post-endpoint endpoint json-rpc-body headers)))