(ns betfairsports.commands
  (:require   [betfairsports.network :as net]
                  [cheshire.core :as cheshire]
                  [clojure.java.io   :as io]))

(defn- args->doc-string
  "Parses the given cmd-map into a doc-string"
  [cmd-map]
  (let [desc (cmd-map :description)
        {{resp-desc :description resp-type :type} :response} cmd-map
        response (str "Response: " resp-type " - " resp-desc)]
    ;; Todo: Add the argument and descriptions
    (str desc response)))

(defn- args->params-vec
  "Parses refspec argument map into simple defn-style parameter vector:
  '[key value & more], etc."
  [args]
  (let [num-non-optional (count (take-while #(not (:optional %)) args))
        num-non-multiple (count (take-while #(not (:multiple %)) args))

        num-fixed        (min num-non-optional (inc num-non-multiple))

        fixed-args       (->> args (map :name) flatten (map symbol) vec)
        has-more? (seq (filter #(:multiple %) args))]
    (if has-more? (conj fixed-args '& 'args) fixed-args)))

(defn- args->opt-bodies
  "Produces a list of optional bodies for the given method name with the given cmd-map args
  (e.g. ([a b] (fn-name a b nil))"
  [fn-name args]
  (let [num-non-optional    (count (take-while #(not (:optional %)) args))
        num-non-multiple     (count (take-while #(not (:multiple %)) args))

        num-fixed                 (min num-non-optional (inc num-non-multiple))
        take-vec-of-args       (fn [count-of-args] (->> args (take count-of-args)
                                                              (map :name) flatten (map symbol) vec))
        vec-of-args                (->> args (map :name) flatten (map symbol) vec)
        call-body                   (fn [fn-name vec-of-args num-nil] (flatten (conj (take num-nil (repeat nil)) (conj (take (- (count vec-of-args) num-nil) vec-of-args) (list fn-name)))))]
    (doall (map (fn [x] (list (take-vec-of-args x) (call-body (symbol fn-name) vec-of-args (- (count vec-of-args) x)))) (range num-fixed (count vec-of-args)))) ))

(defmacro defcommand
  "Defines a function for each method passed by parsing the given cmd-map for
  arguments and doc-string info"
  [method-name cmd-map endpoint debug-mode?]
    (let [args-as-vec           (args->params-vec (cmd-map :args))
          params-hash          (into {} (map #(into [] [(str %) %]) (filter #(not= (symbol "&") %) args-as-vec)))
          opt-bodies-as-vec   (args->opt-bodies method-name (cmd-map :args))
          doc-string              (args->doc-string cmd-map)
          endpoint-method-name (str (cmd-map :method_call_prefix) "/" method-name)]
    (if debug-mode?
      `(println ~method-name ":" \" ~doc-string \"
                "->" ~@opt-bodies-as-vec ~(str args-as-vec))
      `(defn ~(symbol method-name) ~doc-string ~@opt-bodies-as-vec 
          (~args-as-vec
            (net/do-json-rpc ~endpoint ~endpoint-method-name ~params-hash)))
      )
    ;;(intern (symbol (str *ns* "." (name endpoint))) (symbol (name method-name)) (name method-name))
    ))

(defn get-interface-reference
  "Loads the filename specified and parses it as json and parses
  keys into clojure keywords."
  [filename]
  (:methods (cheshire/parse-string (slurp (clojure.java.io/resource filename)) keyword)))

(defmacro defcommands
  "Loads the commands using the specified interface reference json 
  file and calls defcommand for each key."
  ([filename endpoint] `(defcommands ~filename ~endpoint false))
  ([filename endpoint debug-mode?]
     (let [ref (get-interface-reference filename)]
       `(do ~@(map (fn [k v] `(defcommand ~(str (name k)) ~v ~endpoint ~debug-mode?))
                   (keys ref) (vals ref))))))