(ns betfairsports
  (:require [clojure.string :as str]
            [com.jamiei.betfairsports.commands :as commands]
            [com.jamiei.betfairsports.account :as account]
            [com.jamiei.betfairsports.network :as net]
            ))

(def ^{:private true} idds-to-load (atom #{}))

(defmacro with-session 
  "Applies the session-id header to requests executed within the body."
  [session-id & body]
  `(binding [com.jamiei.betfairsports.network/*authentication* ~session-id]
  ~@body))

(defmacro with-app-key 
  "Applies the app-key header to requests executed within the body."
  [app-key & body]
  `(binding [com.jamiei.betfairsports.network/*application* ~app-key]
  ~@body))

(defmacro with-session-and-app-key 
  "Applies the session and app-key headers to requests executed within the body."
  [session-id app-key & body]
  `(with-session ~session-id 
                 (with-app-key ~app-key ~@body)))

(defmacro with-endpoint-override 
  "Executes the body with the endpoint overridden by the provided string. e.g. https://endpoint/json-rpc"
  [endpoint & body]
  `(with-redefs [net/endpoints (fn [endpoint-key] (str endpoint))]
  ~@body))

(defmacro with-connection-manager
  "Executes the body with the reusable-connection-manager options specifed. 
  See https://github.com/dakrone/clj-http#using-persistent-connections for documentation on make-reusable-conn-manager."
  [reusable-conn-manager & body]
  `(binding [com.jamiei.betfairsports.network/*cm* ~reusable-conn-manager]
  ~@body))

(defmacro with-options
  "Executes the body with the options provided overwriting the default clj-http options.
  See https://github.com/dakrone/clj-http for information about the options."
  [options & body]
  `(binding [com.jamiei.betfairsports.network/*options* ~options]
  ~@body))

(defmacro with-error-handling [error-handler & body]
  `(binding [com.jamiei.betfairsports.commands/*handle-error* ~error-handler]
  ~@body))

(defn append-idd-to-load 
  "Adds an IDD file to the list to be loaded when init-idds is called. Does not initialise commands until init is called"
  [res-file-name endpoint-sym]
  (conj idds-to-load [res-file-name endpoint-sym]))

(defn clear-idd-to-load
  "Clears the IDD list to be loaded. Does not clear any initialised commands."
  []
  (reset! idds-to-load #{}))

(defn init-idds 
  "Initialises the commands from the IDDs added to the lists."
  []
  (doall (map #(commands/defcommands (first %) (second %)) idds-to-load)))