(ns if.component.redis
  "Provides a Redis component."
  (:require [com.stuartsierra.component :as component]
            [taoensso.carmine :as car]))

;;;; ===== PROTOCOL =====

(defprotocol RedisClient
  "Methods for interacting with a Redis server"
  (BRPOP [this list-key] [this list-key timeout]
    "Given a list key, blocks the connection and pops an element
    from the tail of the list. If a timeout is given, returns `nil`
    if no element is popped before the timeout is reached.

    Redis: [BRPOP][1]

    [1]: http://redis.io/commands/BRPOP")
  (LLEN [this list-key]
    "Given a list key, returns the length of the list.

    Redis: [LLEN][1]

    [1]: http://redis.io/commands/LLEN")
  (LPUSH [this list-key value]
    "Given a list key and a value, prepends the value to the list.

    Redis: [LPUSH][1]

    [1]: http:/redis.io/commands/LPUSH")
  (PING [this]
    "Returns `PONG` if the connection is still alive.
    Can also be used to measure latency.

    Redis: [PING][1]

    [1]: http:/redis.io/commands/PING")
  (PUBLISH [this channel message]
    "Given a channel name and a message, posts the message to the channel.

    Redis: [PUBLISH][1]

    [1]: http:/redis.io/commands/PUBLISH")
  (listener [this channel callback]
    "Given a channel name and a callback function,
    Subscribes to the channel, passing results to the callback.

    Redis: [SUBSCRIBE][1]

    [1]: http://redis.io/commands/SUBSCRIBE"))


;;;; ===== COMPONENT =====

(defrecord Redis [options spec]
  component/Lifecycle
  (start [this]
    (if spec
      this
      (let [spec-keys [:db :host :password :port :timeout-ms :uri]
            defaults {:host "127.0.0.1" :port 6379}
            client   (as-> (select-keys options spec-keys) spec
                       (apply dissoc spec (for [[k v] spec :when (nil? v)] k))
                       (merge defaults spec)
                       (assoc this :spec spec))]
        (doto client PING))))
  (stop [this]
    (if-not spec
      this
      (assoc this :spec nil)))

  RedisClient
  (BRPOP [this k]
    (BRPOP this k 0))
  (BRPOP [this k timeout]
    (car/wcar this (car/brpop k timeout)))
  (LLEN [this k]
    (car/wcar this (car/llen k)))
  (LPUSH [this k v]
    (car/wcar this (car/lpush k v)))
  (PING [this]
    (car/wcar this (car/ping)))
  (PUBLISH [this k v]
    (car/wcar this (car/publish k v)))
  (listener [this channel f]
    (let [handlers {channel (fn [[k _ v]] (when (= k "message") (f v)))}]
      (car/with-new-pubsub-listener this handlers (car/subscribe channel)))))

(alter-meta! #'->Redis assoc :private true)
(alter-meta! #'map->Redis assoc :private true)


;;;; ===== CONSTRUCTOR =====

(defn new-redis-client
  "Creates and returns a Redis client component from a map of options.
  See [`#'carmine.core/wcar`][1] for options.

  [1]: http://ptaoussanis.github.io/carmine/taoensso.carmine.html#var-wcar"
  [options]
  (map->Redis {:options options}))
