(ns crypto.core
  (:import
    [java.security KeyStore PrivateKey PublicKey Security KeyPair KeyFactory Signature KeyPairGenerator SecureRandom]
    [java.security.cert X509Certificate]
    [java.security.spec X509EncodedKeySpec]
    [java.util Base64]
    [sun.security.x509 X509CertInfo CertificateValidity X500Name CertificateSubjectName CertificateIssuerName CertificateVersion AlgorithmId CertificateAlgorithmId X509CertImpl CertificateVersion CertificateSerialNumber CertificateX509Key]
    [sun.security.pkcs11.wrapper PKCS11Exception]
    [java.util Date]
    [java.security.cert Certificate]
    [java.util Enumeration]
    [javax.crypto Cipher]
    [java.io ByteArrayOutputStream]
    [java.nio ByteBuffer]
    [sun.security.pkcs11 SunPKCS11]))

(defmulti get-key
  (fn [_ {:keys [keystore-type]}] keystore-type))

;only public rsa
;openssl x509 -pubkey -noout -in crt.crt  > pubkey.pem
;remove header, tail and put everything in *one* line
(defmethod get-key :file
  [_ {:keys [file-path]}]
  (let [x509spec (->> file-path
                      slurp
                      clojure.string/trim
                      (.decode (Base64/getDecoder))
                      (X509EncodedKeySpec.))]
    (-> (KeyFactory/getInstance "RSA")
        (.generatePublic x509spec))))

(defn- extract-key [ks ks-password key-type alias]
  (condp = key-type
    :public (-> ks (.getCertificate alias) (.getPublicKey))
    :certificate (.getCertificate ks alias)
    :private (.getKey ks alias (.toCharArray ks-password))
    :key-pair (KeyPair. (extract-key ks ks-password :public alias)
                        (extract-key ks ks-password :private alias))))
;pkcs11 keystore
(defmethod get-key :pkcs11
  [provider {:keys [keystore-password alias key-type]}]
  (let [keystore (KeyStore/getInstance "PKCS11" provider)]
    (do (.load keystore nil (.toCharArray keystore-password))
        (extract-key keystore keystore-password key-type alias))))

(defmethod get-key :pkcs12
  [_ {:keys [keystore-path keystore-password alias key-type]}]
  (with-open [fio (clojure.java.io/input-stream keystore-path)]
    (let [keystore (KeyStore/getInstance "PKCS12")]
      (do (.load keystore fio (.toCharArray keystore-password))
          (extract-key keystore keystore-password key-type alias)))))

(defmethod get-key :jceks
  [_ {:keys [keystore-path keystore-password alias key-type]}]
  (with-open [fio (clojure.java.io/input-stream keystore-path)]
    (let [keystore (KeyStore/getInstance "JCEKS")]
      (do (.load keystore fio (.toCharArray keystore-password))
          (extract-key keystore keystore-password key-type alias)))))

;;These two function are here to avoid the dependence to pallet.thread-expr
;; because this code will also run within the datomic transactor, and all
;; dependencies should be compatible with ancient clojure.
(defn conditional-decode-base64
  [run? data]
  (if run?
    (.decode (Base64/getDecoder) data)
    data))

(defn conditional-encode-base64
  [run? data]
  (if run?
    (.encodeToString (Base64/getEncoder) data)
    data))

(defn crypt
  [{:keys [key-spec provider-name transformation base64-input? base64-output?]}
   cipher-mode
   data]
  (let [provider (Security/getProvider provider-name)
        key (get-key provider key-spec)
        cipher (Cipher/getInstance transformation provider)]
    (do (.init cipher cipher-mode key)
        (->> data
             (conditional-decode-base64 base64-input?)
             (.doFinal cipher)
             (conditional-encode-base64 base64-output?)))))

(defn- build-x509-certificate
  [key-pair x500-owner-name]
  (let [private-key (.getPrivate key-pair)
        from (Date.)
        to (Date. (+ (.getTime from)
                     (* 365 86400000)))
        validity (CertificateValidity. from to) ;one year
        serial-number (BigInteger. 64 (SecureRandom.))
        owner (X500Name. x500-owner-name)
        cert-sign-algorithm (AlgorithmId. AlgorithmId/sha256WithRSAEncryption_oid)

        x509-cert-info (doto (X509CertInfo.)
                         (.set X509CertInfo/VALIDITY validity)
                         (.set X509CertInfo/KEY (CertificateX509Key. (.getPublic key-pair)))
                         (.set X509CertInfo/ALGORITHM_ID (CertificateAlgorithmId. cert-sign-algorithm))
                         (.set X509CertInfo/VERSION (CertificateVersion. CertificateVersion/V3))
                         (.set X509CertInfo/ISSUER owner)
                         (.set X509CertInfo/SERIAL_NUMBER (CertificateSerialNumber. serial-number))
                         (.set X509CertInfo/SUBJECT owner))
        ;; as this is a private package, not part of the Java API there is
        ;; little to none documentation of the signing algorithm name used.
        x509-cert (doto (X509CertImpl. x509-cert-info)
                    (.sign private-key "SHA256withRSA"))
        ;; determine actual signing algorithm
        real-sign-algorithm (.get x509-cert X509CertImpl/SIG_ALG)
        _ (.set x509-cert-info
                (str CertificateAlgorithmId/NAME "."
                     CertificateAlgorithmId/ALGORITHM)
                real-sign-algorithm)]
    (doto (X509CertImpl. x509-cert-info)
      (.sign private-key "SHA256withRSA"))))

(defmulti generate-rsa-keypair!
  (fn [{:keys [keystore-type]} _] keystore-type))

(defmethod generate-rsa-keypair! :pkcs11
  [{:keys [provider-name x500-owner-name keystore-password]} alias]
  (let [hsm-provider (Security/getProvider provider-name)
        keypair-generator (doto (KeyPairGenerator/getInstance "RSA")
                            (.initialize 2048))
        keypair (.generateKeyPair keypair-generator)
        certificate (build-x509-certificate keypair x500-owner-name)]
    (try
      (doto (KeyStore/getInstance "PKCS11" hsm-provider)
        (.load nil (.toCharArray keystore-password))
        (.setKeyEntry alias
                      (.getPrivate keypair)
                      (.toCharArray keystore-password)
                      (into-array X509Certificate [certificate])))
      true
      (catch PKCS11Exception _
        nil))))

(defn jceks-store-certificate
  "stores certificate in a new keystore at the given path"
  [keystore-path keystore-password alias certificate]
    (try
      (with-open [output (clojure.java.io/output-stream keystore-path)]
        (doto (KeyStore/getInstance "JCEKS")
          (.load nil (.toCharArray keystore-password))
          (.setCertificateEntry alias certificate)
          (.store output (.toCharArray keystore-password))))
      true
      (catch Exception e
        (println (.getMessage e))
        nil)))

(defmethod generate-rsa-keypair! :jceks
  [{:keys [keystore-path x500-owner-name keystore-password]} alias]
  (let [keypair-generator (doto (KeyPairGenerator/getInstance "RSA")
                            (.initialize 2048))
        keypair (.generateKeyPair keypair-generator)
        certificate (build-x509-certificate keypair x500-owner-name)
        keystore (KeyStore/getInstance "JCEKS")]
    (try
      (with-open [input (clojure.java.io/input-stream keystore-path)]
        (.load keystore input (.toCharArray keystore-password))
        (.setKeyEntry keystore
                      alias
                      (.getPrivate keypair)
                      (.toCharArray keystore-password)
                      (into-array X509Certificate [certificate])))
      (with-open [output (clojure.java.io/output-stream keystore-path)]
        (.store keystore output (.toCharArray keystore-password)))
      true
      (catch Exception _
        nil))))

(defmulti remove-keypair!
  (fn [{:keys [keystore-type]} _] keystore-type))

(defmethod remove-keypair! :pkcs11
  [{:keys [provider-name keystore-password]} alias]
  (try
    (doto (KeyStore/getInstance "PKCS11"
                                (Security/getProvider provider-name))
      (.load nil (.toCharArray keystore-password))
      (.deleteEntry alias))
    true
    (catch PKCS11Exception _
      nil)))

(defmethod remove-keypair! :jceks
  [{:keys [keystore-path keystore-password]} alias]
  (let [keystore (KeyStore/getInstance "JCEKS")]
    (try
      (with-open [input (clojure.java.io/input-stream keystore-path)]
        (.load keystore input (.toCharArray keystore-password))
        (.deleteEntry keystore alias))
      (with-open [output (clojure.java.io/output-stream keystore-path)]
        (.store keystore output (.toCharArray keystore-password)))
      true
      (catch Exception _
        nil))))

(defn verify
  [{:keys [key-spec provider-name transformation]} ^String data ^String signature]
  (let [provider (Security/getProvider provider-name)
        key (get-key provider key-spec)
        signer (doto (Signature/getInstance transformation provider)
                 (.initVerify key)
                 (.update (.getBytes data)))
        signature-bytes (->> signature
                             (.decode (Base64/getDecoder)))]
    (.verify signer signature-bytes)))

(defn sign
  [{:keys [key-spec provider-name transformation]} ^String data]
  (let [provider (Security/getProvider provider-name)
        key (get-key provider key-spec)
        signer (doto (Signature/getInstance transformation provider)
                 (.initSign key)
                 (.update (.getBytes data)))]
    (->> (.sign signer)
         (.encodeToString (Base64/getEncoder)))))

(defn encrypt
  [params data]
  (crypt params Cipher/ENCRYPT_MODE data))

(defn decrypt
  [params data]
  (crypt params Cipher/DECRYPT_MODE data))

