(ns retroact.swing.jtable
  (:require [clojure.tools.logging :as log]
            [retroact.swing.util :as util])
  (:import (java.awt Dimension)
           (javax.swing JScrollPane JTable RowSorter$SortKey SortOrder SwingUtilities)
           (javax.swing.table AbstractTableModel TableModel TableRowSorter)
           (retroact.swing.compiled.jtable RTableModel)))

; Pass arguments to create-table-model that define how to access the data from the app-value. Also pass the app-value.
; Then set the table model on the JTable. Just create a new table model when the way to access the data changes. But
; if app-value changes, then we should be able to set that somewhere without recreating the table model... maybe using
; ... I don't know.

#_(defn create-table-model []
  (let [model (atom {:data []})]
    (proxy [AbstractTableModel]
           (.setData))))

(defn safe-table-model-set [table f attribute]
  (cond
    (instance? JTable table)
    (let [model (.getModel table)]
      (f model attribute))
    (instance? JScrollPane table) (safe-table-model-set (-> table (.getViewport) (.getView)) f attribute)
    :else (log/error "skipping fn on table because component does not appear to be a table: " table)))

(defn safe-table-set [table f attribute]
  (cond
    (instance? JTable table) (f table attribute)
    (instance? JScrollPane table) (safe-table-set (-> table (.getViewport) (.getView)) f attribute)
    :else (log/error "skipping fn on table because component does not appear to be a table: " table)))

(defn set-table-selection-fn
  "Set the selection function for a JTable. This function will be called when table selection changes.
  Analogous to :tree-selection-fn for JTree."
  [c ctx table-selection-fn]
  (safe-table-model-set c (memfn setSelectionFn table-selection-fn) table-selection-fn))

(defn set-table-render-fn
  "Set the render function for a JTable. This function customizes how cells are rendered.
  Analogous to :tree-render-fn for JTree."
  [c ctx table-render-fn]
  (safe-table-model-set c (memfn setRenderFn table-render-fn) table-render-fn))

(defn set-table-set-value-at-fn
  "Set the function that handles cell value changes when cells are edited.
  This provides a way to handle cell editing events."
  [c ctx set-value-at-fn]
  (safe-table-model-set c (memfn setSetValueAtFn set-value-at-fn) set-value-at-fn))

(defn set-table-get-item-at-fn
  [c ctx table-get-item-at-fn]
  (safe-table-model-set c (memfn setGetItemAtFn table-get-item-at-fn) table-get-item-at-fn))

(defn set-table-auto-resize-mode
  "Set JTable auto-resize mode. Accepts JTable/AUTO_RESIZE_ constants."
  [c ctx mode]
  (safe-table-set c (fn [^JTable t m]
                      (when-let [val m]
                        (.setAutoResizeMode t val)))
                  mode))

(defn set-table-intercell-spacing
  "Set JTable intercell spacing.
  Accepts one or two integers:
  - single integer N: applies to both horizontal and vertical spacing
  - two integers [h v]: horizontal then vertical spacing"
  [c ctx spacing]
  (safe-table-set
    c
    (fn [^JTable t s]
      (when s
        (let [d (cond
                  (number? s) (Dimension. (int s) (int s))
                  (sequential? s)
                  (let [cnt (count s)]
                    (cond
                      (= 1 cnt) (let [n (first s)] (Dimension. (int n) (int n)))
                      (>= cnt 2) (let [[h v] s] (Dimension. (int h) (int v)))
                      :else nil))
                  :else nil)]
          (if d
            (.setIntercellSpacing t d)
            (log/warn "Ignoring :table-intercell-spacing invalid value" s)))))
    spacing))

(defn set-row-selection-allowed [c ctx row-selection-allowed]
  (safe-table-set c (memfn setRowSelectionAllowed row-selection-allowed) row-selection-allowed))

(defn set-column-selection-allowed [c ctx column-selection-allowed]
  (safe-table-set c (memfn setColumnSelectionAllowed column-selection-allowed) column-selection-allowed))

(defn set-table-header
  "This is a \"component\" setter. It takes an actual JTableHeader."
  [c ctx table-header]
  (safe-table-set
    c
    (fn [t h]
      (let [#_sp #_(util/get-parent-scroll-pane t)]
        (.setColumnModel h (.getColumnModel t))
        (.setTableHeader t h)
        #_(.setColumnHeaderView sp h)))
    table-header))

(defn get-table-header [c ctx]
  (let [table (if (instance? JScrollPane c) (.getView (.getViewport c)) c)]
    (.getTableHeader table)))

(defn- ^SortOrder normalize-sort-order [order]
  (cond
    (instance? SortOrder order) order
    (keyword? order)
    (case order
      :asc SortOrder/ASCENDING
      :ascending SortOrder/ASCENDING
      :desc SortOrder/DESCENDING
      :descending SortOrder/DESCENDING
      :unsorted SortOrder/UNSORTED
      SortOrder/UNSORTED)
    :else SortOrder/UNSORTED))

(defn- normalize-sort-spec [spec]
  (cond
    (nil? spec) nil
    ; Single pair [col order]
    (and (vector? spec)
         (= 2 (count spec))
         (number? (first spec)))
    [spec]
    ; Sequence of pairs
    (sequential? spec) (vec spec)
    :else nil))

(defn set-table-sort
  "Configure sorting on a JTable using TableRowSorter.
  Accepts either a single pair [col order] or a sequence of such pairs.
  `order` may be a javax.swing.SortOrder or a keyword one of
  :asc/:ascending, :desc/:descending, or :unsorted.

  Examples:
    {:table-sort [0 SortOrder/ASCENDING]}
    {:table-sort [[2 :desc] [0 :asc]]} ;; secondary sort
  Passing nil clears sorting."
  [c ctx sort-spec]
  (safe-table-set
    c
    (fn [^JTable t spec]
      (let [pairs (normalize-sort-spec spec)]
        (if (nil? pairs)
          (.setRowSorter t nil)
          (let [^TableModel model (.getModel t)
                existing (.getRowSorter t)
                ^TableRowSorter sorter (if (instance? TableRowSorter existing)
                                         existing
                                         (let [s (TableRowSorter. model)]
                                           (.setRowSorter t s)
                                           s))
                sort-keys (mapv (fn [[col order]]
                                   (RowSorter$SortKey. (int col) (normalize-sort-order order)))
                                 pairs)]
            (.setSortKeys sorter sort-keys)
            ; setSortKeys usually triggers a sort; call sort to be explicit
            (.sort sorter)))))
    sort-spec))

; TODO: this doesn't work because it is first run before the table column is created
(defn- set-table-columns- [table columns]
  (let [column-model (.getColumnModel table)
        cm-count (.getColumnCount column-model)
        model-count (.getColumnCount (.getModel table))]
    ; Ensure columns are created from the model when needed (e.g., empty data but headers present).
    (when (< cm-count model-count)
      (.createDefaultColumnsFromModel table))
    (doseq [[n column] columns]
      (let [column-model (.getColumnModel table)]
        (when (and (>= n 0) (< n (.getColumnCount column-model)))
          (let [table-column (.getColumn column-model n)]
            (when-let [preferred-width (cond
                                         (map? column) (get column :preferred-width)
                                         (number? column) column
                                         :else nil)]
              (.setPreferredWidth table-column preferred-width))
            (when-let [max-width (get column :max-width)]
              (.setMaxWidth table-column max-width))
            (when-let [min-width (get column :min-width)]
              (.setMinWidth table-column min-width))))))))

(defn set-table-columns
  [c ctx columns]
  (safe-table-set c set-table-columns- columns))

(defn set-editing-cell
  "Set the editing cell for a JTable. Accepts [row col] or nil to stop editing.
   This will call editCellAt on the JTable to start editing the specified cell."
  [c ctx editing-cell]
  (safe-table-set
    c
    (fn [^JTable t cell-spec]
      (if (nil? cell-spec)
        (when (.isEditing t)
          (let [editor (.getCellEditor t)]
            (when editor
              (.stopCellEditing editor))))
        (let [[row col] cell-spec]
          (when (and (>= row 0) (< row (.getRowCount t))
                     (>= col 0) (< col (.getColumnCount t))
                     (.isCellEditable (.getModel t) row col))
            ; Only change editing if it's different from current
            (let [current-editing-row (.getEditingRow t)
                  current-editing-col (.getEditingColumn t)]
              (when (not (and (= current-editing-row row)
                            (= current-editing-col col)
                            (.isEditing t)))
                ; Use invokeLater to ensure this happens after current rendering
                (SwingUtilities/invokeLater
                  (fn []
                    ; Stop any current editing first
                    (when (.isEditing t)
                      (let [editor (.getCellEditor t)]
                        (when editor
                          (.stopCellEditing editor))))
                    ; Ensure the cell is still valid and editable
                    (when (and (>= row 0) (< row (.getRowCount t))
                               (>= col 0) (< col (.getColumnCount t))
                               (.isCellEditable (.getModel t) row col))
                      ; Scroll to make the cell visible
                      (let [rect (.getCellRect t row col false)]
                        (.scrollRectToVisible t rect))
                      ; Start editing the new cell - try multiple times if needed
                      (loop [attempts 3]
                        (when (and (> attempts 0) (not (.isEditing t)))
                          (.editCellAt t (int row) (int col))
                          (when (not (.isEditing t))
                            (Thread/sleep 10)
                            (recur (dec attempts)))))
                      ; Request focus to ensure the editor is active
                      (when-let [editor-comp (.getEditorComponent t)]
                        (.requestFocus editor-comp))
                      (when (not (.isEditing t))
                        (log/warn "Failed to start editing cell at row" row "col" col)))))))))))
    editing-cell))

(defn create-jtable [{:keys [view]}]
  (let [table-model (RTableModel.)
        table (JTable. ^TableModel table-model)]
    (.setTableComponent table-model table)
    ; TODO: this is a strange decision. Shouldn't it always be in a JScrollPane? Or never? Or perhaps test to see if it is already in one.
    (if (:headers view)
      (JScrollPane. table)
      table)))
