(ns leiningen.release
  "Performs boilerplate tasks when releasing a new version of the project."
  (require [leiningen.core.main :as main]
           (clojure.java [shell :as sh] [io :as io])
           [clojure.string :as s]))

(set! *warn-on-reflection* true)

(def ^:dynamic *config* {})

(defn raise [fmt & args] (throw (RuntimeException. ^String (apply format fmt args))))

(defn sh! [& args]
  (apply println args)
  (let [res (apply sh/sh args)]
    (print (str (:out res) (:err res)))
    (flush)
    (when-not (zero? (:exit res)) (raise "Command failed with exit code %s: %s" (:exit res) args))))

(defn deployment-strategy [project]
  (cond
   (:deploy-via *config*) (:deploy-via *config*)
   (:repositories project) :lein-deploy
   :no-deploy-strategy :none))

(defn ->release [v] (s/replace v #"-SNAPSHOT$" ""))

(defn ->new-snapshot [v]
  (str (s/replace v #"\d+(?!.*\d)" #(str (inc (Long/parseLong %)))) "-SNAPSHOT"))

(defn updated-version [project update-fn]
  (let [curr-version (:version project), new-version (update-fn curr-version)]
    (when (not= new-version curr-version) new-version)))

(defn update-project! [project new-version]
  (if new-version
    (do (println "Edit project.clj: set version to" new-version)
        (-> (slurp "project.clj")
            (s/replace #"(defproject\s+\S+\s+)\"(.+?)\"" (format "$1\"%s\"" new-version))
            (->> (spit "project.clj")))
        (let [p (assoc project :version new-version)]
          (with-meta p (assoc-in (meta p) [:without-profiles :version] new-version))))
    project))

(defmacro lein-do [project & cmds]
  `(do ~@(for [c cmds, cc [`(println "lein" '~c) `(main/apply-task (name '~c) ~project [])]] cc)))

(defn release
  "Cut a new release: bump version, tag, deploy"
  [project & args]
  (binding [*config* (or (:lein-release project) *config*)]
    (let [strat (deployment-strategy project)
          current-version (:version project)
          release-version (updated-version project ->release)
          project (update-project! project release-version)]
      (try
        (println "Deployment strategy is" strat)
        (case strat
          :lein-deploy (lein-do project clean deploy)
          :lein-install (lein-do project clean install)
          :clojars (sh! "scp"
                        (-> project (lein-do clean jar) vals first)
                        (lein-do project pom)
                        "clojars@clojars.org:")
          :none nil
          (raise "Unrecognized deployment strategy: %s" strat))
        (catch Throwable t
          (update-project! project current-version)
          (throw t)))
      (when release
        (sh! "git" "add" "project.clj")
        (sh! "git" "commit" "-m" (format "Release %s (by lein-release)" release)))
      (sh! "git" "tag" (format "%s-%s" (:name project) release))
      (let [new-snapshot (updated-version project ->new-snapshot)]
        (update-project! project new-snapshot)
        (sh! "git" "add" "project.clj")
        (sh! "git" "commit" "-m" (format "Move to %s (by lein-release)" new-snapshot))
        (sh! "git" "push")
        (sh! "git" "push" "--tags")))))
