(ns leiningen.release
  "Performs boilerplate tasks when releasing a new version of the project."
  (require [leiningen.core.main :refer [apply-task]]
           (clojure.java [shell :as sh] [io :as io])
           [clojure.string :as s]))

(set! *warn-on-reflection* true)

(defn raise [fmt & args] (throw (RuntimeException. ^String (apply format fmt args))))

(def ^:dynamic *config* {})

(defn sh! [& args]
  (apply println args)
  (let [res (apply sh/sh args)]
    (doto System/out (.print (:out res)) .flush)
    (doto System/err (.print (:err res)) .flush)
    (when-not (zero? (:exit res)) (raise "Command failed: %s => %s" args res))))

(defn deployment-strategy [project]
  (cond
   (:deploy-via *config*) (:deploy-via *config*)
   (:repositories project) :lein-deploy
   :no-deploy-strategy :lein-install))

(defn ->release-version [v] (s/replace v #"-SNAPSHOT$" ""))

(defn ->next-dev-version [v]
  (str (s/replace v #"\d+(?!.*\d)" #(str (inc (Long/parseLong %)))) "-SNAPSHOT"))

(defn updated-version [project update-fn]
  (let [curr-version (:version project), new-version (update-fn curr-version)]
    (when (not= new-version curr-version) new-version)))

(defn update-project! [project new-version]
  (if new-version
    (do (println "Edit project.clj: set version to" new-version)
        (-> (slurp "project.clj")
            (s/replace #"(defproject\s+\S+\s+)\"(.+?)\"" (format "$1\"%s\"" new-version))
            (->> (spit "project.clj")))
        (let [p (assoc project :version new-version)]
          (with-meta p (assoc-in (meta p) [:without-profiles :version] new-version))))
    project))

(defmacro lein-doto [project & cmds]
  `(do ~@(for [c cmds, cc [`(println "lein" '~c) `(apply-task (name '~c) ~project [])]] cc)))

(defn release [project & args]
  (binding [*config* (or (:lein-release project) *config*)]
    (let [release-version (updated-version project ->release-version)
          project (update-project! project release-version)
          target-dir (:target-path project)
          strat (deployment-strategy project)]
      (println "Deployment strategy" strat)
      (case strat
        :lein-deploy (lein-doto project clean deploy)
        :lein-install (lein-doto project clean install)
        :clojars (do (lein-doto project clean jar pom)
                     (sh! "scp" "pom.xml"
                          (format "%s/%s-%s.jar" target-dir (:name project) release-version)
                          "clojars@clojars.org:"))
        (raise "Unrecognized deployment strategy: %s" strat))
      (when release-version
        (sh! "git" "add" "project.clj")
        (sh! "git" "commit" "-m" (format "Release %s (by lein-release)" release-version))
        (sh! "git" "tag" (format "%s-%s" (:name project) release-version)))
      (let [next-dev-version (updated-version project ->next-dev-version)]
        (update-project! project next-dev-version)
        (sh! "git" "add" "project.clj")
        (sh! "git" "commit" "-m" (format "Move to %s (by lein-release)" next-dev-version))
        (sh! "git" "push")
        (sh! "git" "push" "--tags")))))
