(ns leiningen.release
  "Performs boilerplate tasks when releasing a new version of the project."
  (require [leiningen.core.main :refer [apply-task]]
           (clojure.java [shell :as sh] [io :as io])
           [clojure.string :as s]))

(set! *warn-on-reflection* true)

(defn raise [fmt & args] (throw (RuntimeException. ^String (apply format fmt args))))

(def ^:dynamic *config* {})

(defn sh! [& args]
  (apply println args)
  (let [res (apply sh/sh args)]
    (doto System/out (.print (:out res)) .flush)
    (doto System/err (.print (:err res)) .flush)
    (when-not (zero? (:exit res)) (raise "Command failed: %s => %s" args res))))

(defn deployment-strategy [project]
  (println "deployment strategy" (:deploy-via *config*))
  (cond
   (:deploy-via *config*) (:deploy-via *config*)
   (:repositories project) :lein-deploy
   :no-deploy-strategy :lein-install))

(defn release-version [v] (s/replace v #"-SNAPSHOT$" ""))

(defn next-dev-version [v]
  (str (s/replace v #"\d+(?!.*\d)" #(str (inc (Long/parseLong %)))) "-SNAPSHOT"))

(defn replace-project-version [old-vstring ^String new-vstring]
  (let [^String proj-file (slurp "project.clj")
        new-proj-file (s/replace proj-file
                                 (re-pattern (str "\\(defproject .+? " old-vstring))
                                 new-vstring)
        matcher (.matcher
                 (re-pattern (format "(\\(defproject .+? )\"\\Q%s\\E\"" old-vstring))
                 proj-file)]
    (when-not (.find matcher)
      (raise "Unable to find version string %s in project.clj file." old-vstring))
    (.replaceFirst matcher (format "%s\"%s\"" (.group matcher 1) new-vstring))))

(defn extract-project-version-from-file
  ([] (extract-project-version-from-file "project.clj"))
  ([proj-file]
     (let [s (slurp proj-file)
           m (.matcher (re-pattern "\\(defproject .+? \"([^\"]+?)\"") s)]
       (if-not (.find m)
         (raise "Error: unable to find project version in file: %s" proj-file))
       (.group m 1))))

(defn update-project-version! [project update-fn]
  (let [curr-version (:version project), new-version (update-fn curr-version)]
    (when (not= new-version curr-version)
      (println "Edit project.clj: set version to" new-version)
      (spit "project.clj" (replace-project-version (:version project) new-version))
      new-version)))

(defmacro lein-doto [project & cmds]
  `(do ~@(for [c cmds, cc [`(println "lein" '~c) `(apply-task (name '~c) ~project [])]] cc)))

(defn release [project & args]
  (binding [*config* (or (:lein-release project) *config*)]
    (let [release-version (update-project-version! project release-version)
          project (if release-version
                    (let [p (assoc project :version release-version)]
                      (with-meta p (assoc-in (meta p) [:without-profiles :version] release-version)))
                    project)
          target-dir (:target-path project)
          deployment-strategy (deployment-strategy project)]
      (case deployment-strategy
        :lein-deploy (lein-doto project clean deploy)
        :lein-install (lein-doto project clean install)
        :clojars (do (lein-doto project clean jar pom)
                     (sh! "scp" "pom.xml"
                          (format "%s/%s-%s.jar" target-dir (:name project) release-version)
                          "clojars@clojars.org:"))
        (raise "Unrecognized deployment strategy: %s" deployment-strategy))
      (when release-version
        (sh! "git" "add" "project.clj")
        (sh! "git" "commit" "-m" (str "lein-release: " release-version))
        (sh! "git" "tag" (format "%s-%s" (:name project) release-version)))
      (update-project-version! project next-dev-version)
      (sh! "git" "add" "project.clj")
      (sh! "git" "commit" "-m" (str "lein-release: " next-dev-version))
      (sh! "git" "push")
      (sh! "git" "push" "--tags"))))
