(ns dirwatch.core
  (:require [clojure.core.async :refer (chan >!! close! thread)] 
            [taoensso.timbre :refer (log trace debug info error fatal)])
  (:import (java.io File) 
           (java.nio.file WatchService
                          FileSystems
                          Files
                          Path
                          Paths
                          FileVisitor
                          FileVisitResult
                          WatchKey
                          Watchable
                          WatchEvent
                          WatchEvent$Kind
                          StandardWatchEventKinds
                          LinkOption
                          ClosedWatchServiceException)))

(def ^:private event-kinds {StandardWatchEventKinds/ENTRY_CREATE :create
                            StandardWatchEventKinds/ENTRY_MODIFY :modify
                            StandardWatchEventKinds/ENTRY_DELETE :delete})

(defn path
  [first & more]
  (let [f (str first)
        m (map str more)]
    (. Paths get f (into-array String m))))

(defn dir?
  [^Path path]
  (. Files isDirectory path (into-array LinkOption [])))

(defn- update
  [m k v & kvs]
  (apply swap! m assoc k v kvs))

(defn- emit
  [ch kind path]
  (debug "Event" (str kind)  (str path))
  (>!! ch {:kind kind :path path}))

(defn- watcher-visitor
  [ws & more]
  (let [f (first more)
        callback #(when f (f %))]   
    (reify FileVisitor 
      (preVisitDirectory [_ dir _]
        FileVisitResult/CONTINUE) 
      (postVisitDirectory [_ dir err] 
        (when err (throw err))
        (. dir (register ws (into-array (keys event-kinds))))
        (callback dir)
        FileVisitResult/CONTINUE)
      (visitFile [_ file _]
        (callback file)
        FileVisitResult/CONTINUE)
      (visitFileFailed [_ file err]
        (throw err)))))

(defn- poll-event
  [^WatchService ws ch]
  (let [^WatchKey key (.. ws take)]
    (let [events (. key pollEvents)
          ^Path w (. key watchable)
          ^Path dir (. w toAbsolutePath)]
      (doseq [^WatchEvent event events]
        (let [kind (get event-kinds (. event kind))
              ^Path context (. event context)
              path (. dir resolve context)]
          (if
            (and (= kind :create) (dir? path))
            (Files/walkFileTree path (watcher-visitor ws #(emit ch :create %)))
            (emit ch kind path))))
      (. key reset))))

(defprotocol Watcher
  (start [this])
  (stop [this])
  (close [this]))

(defrecord DirWatcher [^WatchService ws stat]
  Watcher
  (start [this]
    (when-not (:chan @stat)
      (debug "Start to watch" (str (:dir @stat)))
      (update stat :chan (chan))
      (thread 
        (try
          (while (:chan @stat)
            (poll-event ws (:chan @stat)))
          (catch ClosedWatchServiceException e
            (trace "WatchService has been closed"))
          (catch Exception e
            (fatal e "Failed to watch directory")))))
    (:chan @stat))
  (stop [this] 
    (debug "Stop to watch" (str (:dir @stat)))
    (println (close! (:chan @stat)))
    (update stat :chan nil))
  (close [this]
    (when
      (:chan @stat)
      (stop this))
    (. ws close)))

(defmulti watch (fn [dir] (class dir)))
(defmethod watch String [dir]
  (watch (path dir)))
(defmethod watch Path [dir]
  (let [ws (.. FileSystems getDefault newWatchService)]
    (Files/walkFileTree dir (watcher-visitor ws))
    (DirWatcher. ws (atom {:dir dir}))))

