(ns hypercrud.compile.eval
  (:require [cats.monad.either :as either]
            [cljs.analyzer :as analyzer]
            [cljs.tagged-literals :as tags]
            [cljs.js :as cljs]
            [hypercrud.client.readers :as client-readers]
            [hypercrud.readers :as hc-readers]
            [hypercrud.types.DbVal :refer [read-DbVal]]
            [hypercrud.types.EntityRequest :refer [read-EntityRequest]]
            [hypercrud.types.Err :refer [read-Err]]
            [hypercrud.types.QueryRequest :refer [read-QueryRequest]]
            [markdown.core]))


(def eval-str-
  (memoize (fn [code-str]
             {:pre [(not (empty? code-str))]}
             ;; Hack - we don't understand why cljs compiler doesn't handle top level forms naturally
             ;; but wrapping in identity fixes the problem
             (let [code-str' (str "(identity\n" code-str "\n)")]
               (binding [analyzer/*cljs-warning-handlers* []
                         tags/*cljs-data-readers* (merge tags/*cljs-data-readers*
                                                         {'entity hc-readers/entity
                                                          'URI client-readers/URI
                                                          'hypercrud.types.DbVal.DbVal read-DbVal
                                                          'hypercrud.types.EntityRequest.EntityRequest read-EntityRequest
                                                          'hypercrud.types.Err.Err read-Err
                                                          'hypercrud.types.QueryRequest.QueryRequest read-QueryRequest

                                                          ; deprecated
                                                          ; we no longer serialize to `->entity`but we need to still support reading from it
                                                          '->entity hc-readers/->entity})]
                 (cljs/eval-str (cljs/empty-state)
                                code-str'
                                nil
                                {:eval cljs/js-eval}
                                identity))))))

(defn eval-str-and-throw [code-str]
  (let [compiler-result (eval-str- code-str)]
    (if-let [error (:error compiler-result)]
      (throw error)
      (:value compiler-result))))

(def eval-
  (memoize (fn [form]
             (let [form' `(identity ~form)]
               (binding [analyzer/*cljs-warning-handlers* []]
                 (cljs/eval (cljs/empty-state)
                            form'
                            {:eval cljs/js-eval}
                            identity))))))

(defn wrap-from-compiler-result [eval-result input]
  (let [{value :value error :error} eval-result]
    (cond
      error (either/left {:message "cljs eval failed" :data {:cljs-input input :cljs-result eval-result}})
      :else (either/right value))))

(defn eval-str' [code-str]
  ;if there is a string rep in the meta, the object itself is code
  (if (:str (meta code-str))
    (either/right code-str)
    (-> code-str eval-str- (wrap-from-compiler-result code-str))))

; todo im dead code
(defn eval' [form]
  (-> form eval- (wrap-from-compiler-result form)))

(defn validate-user-code-str [code-str]
  (cond
    (:str (meta code-str)) code-str
    (and (string? code-str) (not (empty? code-str))) code-str
    (and (string? code-str) (empty? code-str)) nil          ; coerce empty to nil
    (not= nil code-str) code-str
    :else nil))
