(ns ant-man.helpers
  (:require [montoux.codec.core :as codec]))

(defonce ^:private transcoder codec/base64)

(defn read-key
  "Helper funtion that reads and parses meta information from `keyify`"
  [x]
  (-> x (meta) :key (codec/decode-edn transcoder) :payload))

(defn- default-key-fn [x]
  (codec/encode-edn {:payload x} transcoder))

(defn keyify
  "Given a function and collection will add unique `:key` value
  to metadata information

  Customizable `:key-fn` functionality for specifying list item `:key` value

  Use example:

  ```
  (require '[ant-man.core :as ant-man]
           '[ant-man.helpers :as ant-man-helpers]
           '[reagent.core :as r])

  (def selection (r/atom nil))
  (def example-collection [{:label \"foo\" :id 1} {:label \"bar\" :id 2}])

  ;; Default
  [ant-man/select {:on-change (fn [val] (reset! selection val))
                   :value @selection}
    (ant-man-helpers/keyify
      (fn [{:keys [label id]}]
        [ant-man/select-option {:value id} label])
      example-collection)]

  ;; Custom `:key-fn`
  [ant-man/select {:on-change (fn [val] (reset! selection val))
                   :value @selection}
    (ant-man-helpers/keyify
      (fn [{:keys [label id]}]
        [ant-man/select-option {:value id} label] :key-fn (fn [x] [(inc x)]))
      example-collection)]
  ```"
  [f xs & {:keys [key-fn] :or {key-fn default-key-fn}}]
  (let [attach-metadata (fn [x]
                          (with-meta (f x)
                            {:key (key-fn x)}))]
    (map attach-metadata xs)))
