package com.hbase.haxwell;

import com.google.common.collect.ImmutableMap;
import org.apache.hadoop.hbase.CoprocessorEnvironment;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.coprocessor.BaseRegionObserver;
import org.apache.hadoop.hbase.coprocessor.ObserverContext;
import org.apache.hadoop.hbase.coprocessor.RegionCoprocessorEnvironment;
import org.apache.hadoop.hbase.regionserver.wal.WALEdit;
import org.apache.hadoop.hbase.util.JsonMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Map;

/**
 * @author phaneesh
 */
public class HaxwellRegionObserver extends BaseRegionObserver {

    private static final Logger log = LoggerFactory.getLogger(HaxwellRegionObserver.class);

    private KafkaMessageProducer messageProducer;

    private String topic;

    private boolean enableDelete;

    public void start(CoprocessorEnvironment e) throws IOException {
        final String brokers = e.getConfiguration().get("haxwell.kafka.bootstrap.servers");
        final String acks = e.getConfiguration().get("haxwell.kafka.acks", "all");
        final int retries = e.getConfiguration().getInt("haxwell.kafka.retries", 3);
        final int batchSize = e.getConfiguration().getInt("haxwell.kafka.batch.size", 1);
        final int autoCommitInterval = e.getConfiguration().getInt("haxwell.kafka.autocommit.interval.ms", 30000);
        this.topic = e.getConfiguration().get("haxwell.kafka.topic");
        this.enableDelete = e.getConfiguration().getBoolean("haxwell.delete.enable", false);
        this.messageProducer = new KafkaMessageProducer(brokers, acks, retries, batchSize, autoCommitInterval);
    }

    @Override
    public void postPut(final ObserverContext<RegionCoprocessorEnvironment> e,
                        final Put put, final WALEdit edit, final Durability durability) throws IOException {
        sendData(e, put.getId(), put.toMap(Integer.MAX_VALUE), "PUT");
    }

    @Override
    public void postDelete(final ObserverContext<RegionCoprocessorEnvironment> e,
                           final Delete delete, final WALEdit edit, final Durability durability) throws IOException {
        if (enableDelete) {
            sendData(e, delete.getId(), delete.toMap(Integer.MAX_VALUE), "DELETE");
        }
    }


    private void sendData(final ObserverContext<RegionCoprocessorEnvironment> e, final String id,
                          final Map<String, Object> data, final String operation) throws IOException {
        final String tableName = e.getEnvironment().getRegion().getTableDesc().getNameAsString();
        if (log.isDebugEnabled()) {
            log.debug("Table Name: {} | Data: {}", tableName, JsonMapper.writeMapAsString(data));
        }
        try {
            messageProducer.send(topic, id, JsonMapper.writeMapAsString(
                    ImmutableMap.<String, Object>builder()
                            .put("table", tableName)
                            .put("operation", operation)
                            .put("id", id)
                            .put("data", data)
                            .build()
            ));
        } catch (Exception ex) {
            log.error("Error sending event to Kafka", ex);
            throw new IOException(ex);
        }
    }

}
