package com.hbase.haxwell;

import org.apache.hadoop.metrics.MetricsContext;
import org.apache.hadoop.metrics.MetricsUtil;
import org.apache.hadoop.metrics.Updater;
import org.apache.hadoop.metrics.util.*;
import org.apache.hadoop.metrics.MetricsRecord;

import javax.management.ObjectName;


public class HaxwellMetrics implements Updater {

    private final String recordName;
    private final MetricsRegistry metricsRegistry;
    private final MetricsRecord metricsRecord;
    private final MetricsContext context;
    private final HaxwellMetricsMXBean mbean;

    private final MetricsTimeVaryingRate processingRate;

    private final MetricsLongValue lastTimestampInputProcessed;

    public HaxwellMetrics(String recordName) {
        this.recordName = recordName;
        metricsRegistry = new MetricsRegistry();
        processingRate = new MetricsTimeVaryingRate("haxwellProcessed", metricsRegistry);
        lastTimestampInputProcessed = new MetricsLongValue("lastHaxwellTimestamp", metricsRegistry);

        context = MetricsUtil.getContext("repository");
        metricsRecord = MetricsUtil.createRecord(context, recordName);
        context.registerUpdater(this);
        mbean = new HaxwellMetricsMXBean(this.metricsRegistry);
    }

    public void shutdown() {
        context.unregisterUpdater(this);
        mbean.shutdown();
    }

    /**
     * Report that a filtered SEP operation has been processed. This method should only be called to
     * report SEP operations that have been processed after making it through the filtering process.
     *
     * @param duration The number of millisecods spent handling the SEP operation
     */
    public void reportFilteredSepOperation(long duration) {
        processingRate.inc(duration);
    }

    /**
     * Report the original write timestamp of a SEP operation that was received. Assuming that SEP
     * operations are delivered in the same order as they are originally written in HBase (which
     * will always be the case except for when a region split or move takes place), this metric will always
     * hold the write timestamp of the most recent operation in HBase that has been handled by the SEP system.
     *
     * @param writeTimestamp The write timestamp of the last SEP operation
     */
    public void reportSepTimestamp(long writeTimestamp) {
        lastTimestampInputProcessed.set(writeTimestamp);
    }

    @Override
    public void doUpdates(MetricsContext unused) {
        synchronized (this) {
            for (MetricsBase m : metricsRegistry.getMetricsList()) {
                m.pushMetric(metricsRecord);
            }
        }
        metricsRecord.update();
    }

    public class HaxwellMetricsMXBean extends MetricsDynamicMBeanBase {
        private final ObjectName mbeanName;

        public HaxwellMetricsMXBean(MetricsRegistry registry) {
            super(registry, "Haxwell Metrics");
            mbeanName = MBeanUtil.registerMBean("haxwell", recordName, this);
        }

        public void shutdown() {
            if (mbeanName != null)
                MBeanUtil.unregisterMBean(mbeanName);
        }
    }
}
