package com.hbase.haxwell;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.hbase.haxwell.api.HaxwellEvent;
import com.hbase.haxwell.api.HaxwellEventListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;

public class HaxwellEventExecutor {

    private Log log = LogFactory.getLog(getClass());
    private HaxwellEventListener eventListener;
    private int numThreads;
    private int batchSize;
    private HaxwellMetrics HaxwellMetrics;
    private List<ThreadPoolExecutor> executors;
    private Multimap<Integer, HaxwellEvent> eventBuffers;
    private List<Future<?>> futures;
    private HashFunction hashFunction = Hashing.murmur3_32();
    private boolean stopped = false;

    public HaxwellEventExecutor(HaxwellEventListener eventListener, List<ThreadPoolExecutor> executors, int batchSize, HaxwellMetrics HaxwellMetrics) {
        this.eventListener = eventListener;
        this.executors = executors;
        this.numThreads = executors.size();
        this.batchSize = batchSize;
        this.HaxwellMetrics = HaxwellMetrics;
        eventBuffers = ArrayListMultimap.create(numThreads, batchSize);
        futures = Lists.newArrayList();
    }

    /**
     * Schedule a {@link HaxwellEvent} for execution.
     * <p>
     * The event will be buffered until it can be executed within a batch of the configured batch size, or until the
     * {@link #flush()} method is called.
     *
     * @param HaxwellEvent event to be scheduled
     */
    public void scheduleHaxwellEvent(HaxwellEvent HaxwellEvent) {

        if (stopped) {
            throw new IllegalStateException("This executor is stopped");
        }

        // We don't want messages of the same row to be processed concurrently, therefore choose
        // a thread based on the hash of the row key
        int partition = (hashFunction.hashBytes(HaxwellEvent.getRow()).asInt() & Integer.MAX_VALUE) % numThreads;
        List<HaxwellEvent> eventBuffer = (List<HaxwellEvent>)eventBuffers.get(partition);
        eventBuffer.add(HaxwellEvent);
        if (eventBuffer.size() == batchSize) {
            scheduleEventBatch(partition, Lists.newArrayList(eventBuffer));
            eventBuffers.removeAll(partition);
        }
    }

    private void scheduleEventBatch(int partition, final List<HaxwellEvent> events) {
        Future<?> future = executors.get(partition).submit(new Runnable() {
            @Override
            public void run() {
                try {
                    long before = System.currentTimeMillis();
                    log.debug("Delivering message to listener");
                    eventListener.processEvents(events);
                    HaxwellMetrics.reportFilteredSepOperation(System.currentTimeMillis() - before);
                } catch (RuntimeException e) {
                    log.error("Error while processing event", e);
                    throw e;
                }
            }
        });
        futures.add(future);
    }

    /**
     * Flush all buffered HaxwellEvent batches, causing them to be started up for execution.
     * <p>
     * Returns all {@code Future}s for all events that have been scheduled since the last time this method was called.
     */
    public List<Future<?>> flush() {
        for (int partition : eventBuffers.keySet()) {
            List<HaxwellEvent> buffer = (List<HaxwellEvent>)eventBuffers.get(partition);
            if (!buffer.isEmpty()) {
                scheduleEventBatch(partition, Lists.newArrayList(buffer));
            }
        }
        eventBuffers.clear();
        List<Future<?>> flushedFutures = Lists.newArrayList(futures);
        return flushedFutures;
    }
}
