(ns ligable.plugins
  "A few default plugins for ligable and tagged literal expansions")

(defn default-reader
  "This function keeps the structure of tagged literals intact so it works
  well as the *default-data-reader-fn* for plugins."
  [tag arg]
  (symbol (str "#" tag " " arg)))

(defn reader [string rdrs]
  (binding [*data-readers* rdrs
            *default-data-reader-fn* default-reader]
    (read-string string)))

(defn key-to-str [& ks]
  (apply (comp #(subs % 1) str) ks))

(defn key-to-sym [& ks]
  (apply (comp symbol key-to-str) ks))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Data Plugin

(defn data-plugin
  "The data-plugin creates Reagent atoms for each keyword given."
  [string]
  (let [data-map# (reader string {})
        data# (reduce-kv (fn [acc# k# v#]
                           (conj acc# (key-to-sym k#)
                             `(reagent.core/atom ~v#)))
                [] data-map#)]
    [:let data#]))

(comment (data-plugin "{:homepage-list \"lots o data\"}"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; UI Plugin

(defn page-atom
  "This is the atom used for page routing"
  [homepage]
  `(reagent.core/atom {:page ~homepage}))

;; NOTE: assumes we're using hiccup syntax
(defn- page-screen
  "Used for the page dispatch based on the value of page-atom"
  [ui]
  `(fn []
     (let [page# (:page (deref ~'patom#))
           current# (case page#
                      ~@(reduce (fn [acc# [p# _]]
                                  (conj acc# (key-to-str p#)
                                    [(key-to-sym p# "#")]))
                          [] ui))]
       [:div current#])))

(comment (let [patom (page-atom :home)]
            (page-screen {:home "woot" :stuff "stuff"} patom)))

;; FIXME: there should be a better way than using :homepage + :mount but
;;  this is pretty good for now because it keeps the UI details in the
;;  UI section
(defn ui-plugin
  "This builds a UI with app pages, a page router, and DOM code for each page.
  A default page of :homepage is required and acts as the first page that
  gets loaded (the homepage/homescreen)."
  [string]
  (let [page# (fn [page#]
                ;; NOTE: niling the page-atom here ensures re-rendering
                `(fn [e#]
                   (do (reset! ~'patom# nil)
                       (reset! ~'patom# {:page ~(key-to-str page#)}))))
        literals#  {'page page#}
        data# (reader string literals#)
        homepage# (-> data# (get-in [:opts :homepage]) key-to-str)
        mount# (get-in data# [:opts :mount])
        patom# (page-atom homepage#)
        new-map# (dissoc data# :opts)
        pscreen# (page-screen new-map#)
        ui# `(let [~'patom# ~patom#
                   ~@(reduce (fn [acc [p# body#]]
                               (conj acc
                                 (key-to-sym p# "#")
                                 `(fn [] ~body#)))
                       [] new-map#)
                   screen# ~pscreen#]
               (reagent.core/render-component [screen#]
                 (.getElementById js/document ~mount#)))]
    [:out ui#]))


(comment (clojure.pprint/pprint (ui-plugin
                                   "{:home [:section [:h1 {:icons [\"compose\" \"search\"]} \"woot\"] [:ul {:data #bind [:data :homepage-list]} [:li [:a {:on-click #page :prefs} \"Prefs\"]]]], :prefs [:section [:h1 \"Prefences\"] [:p {:on-click #page :home} (+ 1 2)]], :opts {:homepage :home, :mount \"app\"}}",)))
