(ns ligable.core
  (:require [ligable.plugins :as lp]))

;; Custom tagged literals must be local to the plugin because there's no
;; way to tell which plugin will get expanded first.

(defmulti ligable-plugin
  "Executes the ligable plugins. The defaults are :data and :ui.
  Other plugins can be added by registering new methods."
  (fn [name code] (vec name)))

(defmethod ligable-plugin [:data] [_ code] (lp/data-plugin code))

(defmethod ligable-plugin [:ui] [_ code] (lp/ui-plugin code))


;; steps for the compiler:
;; 1. expand plugin code
;; 2. use an output language when rebinding each plugin to determines how
;;  the plugin will be output into cljs runtime. Return a map with
;;  the output lang as the top-level keys in place of the plugin keys
;; 3. expand the bind/reset literals, ensuring that data is bound with
;;  the expanded sexprs (like reagent atoms from the data plugin)
;; 4. write the output based on the output language

;; let - Merge all of the let elements and dump them into the binding of
;;         the let expr
;; out - the :out value is dumped into runtime and is wrapped by the let
;;         expression(all :out exprs are merged inside the parent let)
;; none - no output
;;
;; output lang:
;; let ::== :let
;; out ::== :out
;; none ::== !(:let | :out)

(defn merge-outs
  "Separates the output based on the output lang tags"
  [out-vec]
  (let [values# (fn [coll]
                  (reduce (fn [acc# [_ v#]]
                            (conj acc# v#))
                    [] coll))
        g# (group-by (comp identity first) out-vec)
        lets# (first (values# (:let g#)))
        outs# (values# (:out g#))]
    {:let lets# :out outs#}))

(defn write-output
  "Formats the code for cljs consumption"
  [str-vec]
  (let [clj-map# (lp/reader str-vec {})
        outs (merge-outs clj-map#)]
    `(let [~@(:let outs)]
       ~@(:out outs))))

(defn- plugin-expansion
  "Expands all of the plugins and their tagged literals"
  [lmap]
  (let [new-map# (reduce-kv (fn [acc# k# v#]
                              (str acc# (ligable-plugin [k#] v#)))
                   "" lmap)
        ;; wrap it so the entire map gets read
        new-str# (str "[" new-map# "]")]
    (write-output new-str#)))

(comment (clojure.pprint/pprint
            (plugin-expansion
              {:ui
               "{:home [:section [:h1 {:icons [\"compose\" \"search\"]} \"woot\"] [:ul {:data #bind [:data :homepage-list]} [:li [:a {:on-click #page :prefs} \"Prefs\"]]]], :prefs [:section [:h1 \"Prefences\"] [:p {:on-click #page :home} (+ 1 2)]], :opts {:homepage :home, :mount \"app\"}}",
               :data "{:homepage-list \"lots o data\"}"})))


(defn default-expansion
  "Expands the default #bind/#reset tagged literals"
  [unexp ligstr]
  (let [bind# (fn [path#]
                (get-in unexp path#))
        reset# (fn [path# val#]
                 (assoc-in unexp path# val#))
        readers#  {'bind bind#
                   'reset reset#}]
    (lp/reader ligstr readers#)))

(defn ligand* [ligstr]
  (let [unexpanded# (lp/reader ligstr {})
        exp# (default-expansion unexpanded# ligstr)
        lmap# (reduce-kv (fn [acc# k# v#]
                           (assoc acc# k# (str v#)))
                {} exp#)]
    (plugin-expansion lmap#)))

(defmacro ligand
  "A ligand is a declarative, map of how your app will work. This function
  takes a ligand as a string while ligand-file requires an external file.
  Once processed, the ligand is expanded into the namespace that calls it
  and may use any of the declared namespace dependencies.

  The ligand structure is a map with plugin names as keys and the values for
  the respective plugins as the values. The default plugins provided are :ui
  for UI code and :data for Reagent atoms of data. The UI code is normal
  Reagent code with hiccup syntax. Custom plugins may be added by expanding
  the ligable-plugin multimethod with a new method. A plugin may also
  declare its own tagged literals, however, they will be expanded after
  #bind and #reset."
  [string]
  (ligand* string))

(defmacro ligand-file
  "Builds a ligand from a local file."
  [file]
  (let [s# (slurp file)]
    (ligand* s#)))

(comment (clojure.pprint/pprint
            (macroexpand '(ligand-file "test/example-ligand.cljs"))))
