(ns ligable.core
  (:require [ligable.plugins :as lp]))

(defmulti ligable-plugin
  "Executes the ligable plugins. The defaults are :data and :ui.
  Other plugins can be added by registering new methods."
  (fn [name code] (vec name)))

(defmethod ligable-plugin [:data] [_ code] (lp/data-plugin code))

(defmethod ligable-plugin [:ui] [_ code] (lp/ui-plugin code))


(defn- plugin-expansion [coll mount]
  `(do (declare ~'page-atom)
       ~@(reduce (fn [acc [k v]]
                   (into acc (ligable-plugin [k] v)))
           '() coll)
       (reagent.core/render-component [~'screen]
         (.getElementById js/document ~mount))))

(comment (clojure.pprint/pprint (plugin-expansion {:ui {:home [:section [:stuff]]
                                                         :prefs [:moar]
                                                         :homepage :home}
                                                    :data {:homepage-list []}} "")))

;; FIXME: I know defining page-atom in the macro is really bad form
;; FIXME: the #page macro makes the UI plugin completely terrible (replace
;;  with something better). It smears the UI code into everything.
(defmacro ligand
  "A ligand is a declarative, map of how your app will work. It's written in
  an external file but is expanded into the namespace that calls it.

  file -- the file with your app's description
  mount -- a mount point in the DOM

  The ligand structure is a map with plugin names as keys and the values for
  the respective plugins as the values. The default plugins provided are :ui
  for UI code and :data for Reagent atoms of data. The UI code is normal Reagent
  code with hiccup syntax. Custom plugins may be added by expanding the
  ligable-plugin multimethod with a new method.

  Reserves the variable page-atom for routing and each page name.
  Don't override these names."
  [file mount]
  (let [txt# (slurp file)
        ;; FIXME: wimpy assertion but should work for most cases
        home-check# (try (re-seq #" :homepage " txt#)
                         (catch Error e (str "Ligable Error: no :homepage given for the UI plugin.")))
        unexpanded# (binding [*default-data-reader-fn*
                              (fn [tag arg]
                                (symbol (str "#" tag " " arg)))]
                      (read-string txt#))
        bind# (fn [path#]
                (get-in unexpanded# path#))
        reset# (fn [path# val#]
                 (assoc-in unexpanded# path# val#))
        page# (fn [page#]
                ;; NOTE: niling the page-atom here ensures re-rendering
                `(fn [~'e]
                   (do (reset! ~'page-atom nil)
                       (reset! ~'page-atom {:page ~(lp/key-to-str page#)}))))
        readers#  {'bind bind#
                   'reset reset#
                   'page page#}
        data#  (binding [*data-readers* readers#]
                      (read-string txt#))]
    (plugin-expansion data# mount)))


(comment (clojure.pprint/pprint (macroexpand '(ligand "test/example-ligand.cljs" "app"))))
