(ns omniconf.ssm
  (:require [amazonica.aws.simplesystemsmanagement :as ssm]
            [omniconf.core :as cfg]))

(defn- ssm-key->key-vector
  "Transform an SSM key (which is a string) into the vector of keywords, split on
  separator. Don't include the hierarchy prefix (SSM path)."
  [^String ssm-key, ^String prefix, hierarchy-separator]
  (when (not= (.indexOf ssm-key prefix) 0)
    (throw (ex-info (format "SSM key does not start with the prefix: %s - %s"
                            ssm-key prefix) {})))
  (let [^String key (subs ssm-key (count prefix))]
    (->> (.split key hierarchy-separator)
         (remove empty?)
         (mapv keyword))))

#_(ssm-key->key-vector "/foo/bar/baz/qux" "/foo/bar" "/")

(defn- get-parameters
  "Fetches parameters from SSM recursively under the given `path`. Returns a map
  of keys to values, where keys are already split into vectors of keywords by
  the `separator`."
  [path separator]
  (let [parameters (ssm/get-parameters-by-path {:path path
                                                :recursive true
                                                :with-decryption true})]
    (->> (:parameters parameters)
         (map (fn [{:keys [name value]}]
                [(ssm-key->key-vector name path separator) value]))
         (into {}))))

(defn set-value-from-ssm
  "Fetch a single value from Amazon SSM by the given `ssm-key-name` and set it in
  Omniconf by `omniconf-key`."
  [omniconf-key ssm-key-name]
  (let [value (-> (ssm/get-parameter {:name ssm-key-name :with-decryption true})
                  :parameter :value)]
    (cfg/set omniconf-key value)))

(defn populate-from-ssm
  "Fill configuration from AWS Systems Manager. Recursively look up all parameters
  under the given `path`."
  [path]
  (try
    (let [path path
          parameters (get-parameters path "/")]
      (when (empty? parameters)
        (@@#'cfg/logging-fn "WARNING: No parameters received from SSM:" path))
      (let [scheme (#'cfg/flatten-and-transpose-scheme :kw @@#'cfg/config-scheme)]
        (doseq [[key value] parameters
                :let [spec (get scheme key)]
                :when (and spec (not (:nested spec)))]
          (cfg/set key value))))
    (catch clojure.lang.ExceptionInfo e (#'cfg/log-and-rethrow e))))
