(ns remote-fs.adapter.local
  ^{:author "Jingcheng Yang<yjcyxky@163.com>"
    :description "Clojure Wrapper around local filesystem"}
  (:require [java-time :as t]
            [clojure.java.io :as io]
            [clojure.string :as clj-str])
  (:import (java.nio.file Files
                          LinkOption
                          NoSuchFileException
                          Path)
           (java.io File)))

(defn coerce-path-to-string
  "Convert a path to a string, the path may be a Path object or a string object.

   ```clojure
   (coerce-path-to-string (as-path \"/Users/codespace/\" \"test.txt\"))
   (coerce-path-to-string \"/Users/codespace/test.txt\")
   ```
  "
  ^String [path]
  (if (instance? Path path)
    (str path)
    path))

(defn as-file
  "Convert a string to a File object.

   ```clojure
   (as-file \"/Users/codespace/test.txt\")
   (as-file \"/Users/codespace/\" \"test.txt\")
   ```
  "
  ^File [^String path & paths]
  (apply io/file (coerce-path-to-string path) (map coerce-path-to-string paths)))

(defn join-paths
  "Joins one or more path segments into a single String path object."
  ^Path [^String path & paths]
  (.getPath (apply as-file path paths)))

(defn as-path
  "Join path and convert it to a FileSystem Path object.

   ```clojure
   ;; Output: #object[sun.nio.fs.UnixPath 0x3379968a \"/Users/codespace/test.txt\"]
   (as-path \"/Users/codespace/\" \"test.txt\")
   ```
  "
  ^Path [^String path & paths]
  (.toPath ^File (apply as-file path paths)))

(defn ->link-options
  "Converts a hash-map of options into an array of LinkOption objects.
   
   | key              | description |
   | -----------------|-------------|
   | `:nofollow-links`| Adds LinkOption/NOFOLLOW_LINKS to the array. Default: `false`|"
  (^"[Ljava.nio.file.LinkOption;" []
   (make-array LinkOption 0))
  (^"[Ljava.nio.file.LinkOption;" [{:keys [nofollow-links]}]
   (into-array
    LinkOption
    (if nofollow-links
      [LinkOption/NOFOLLOW_LINKS]
      []))))

(defn directory?
  "Return true if `path` is a directory."
  ([path]
   (directory? path {}))
  ([path {:keys [nofollow-links]}]
   (Files/isDirectory (as-path path) (->link-options {:nofollow-links nofollow-links}))))

(defn regular-file?
  "Return true if `path` is a regular file (not a soft link)."
  ([path]
   (regular-file? path {}))
  ([path {:keys [nofollow-links]}]
   (Files/isRegularFile (as-path path) (->link-options {:nofollow-links nofollow-links}))))

(defn- exists?
  "Return true if `path` exists."
  ([path]
   (exists? path {}))
  ([path {:keys [nofollow-links]}]
   (Files/exists (as-path path) (->link-options {:nofollow-links nofollow-links}))))

(defn get-attribute
  ([path attribute]
   (get-attribute path attribute {}))
  ([path attribute {:keys [nofollow-links]}]
   (Files/getAttribute (as-path path)
                       attribute
                       (->link-options {:nofollow-links nofollow-links}))))

(defn creation-time
  "Get the creation time of a file or directory."
  ([path]
   (creation-time path {}))
  ([path {:keys [nofollow-links]}]
   (get-attribute path "basic:creationTime" {:nofollow-links nofollow-links})))

(defn last-modified-time
  "Get the last modified time of a file or directory."
  ([path]
   (last-modified-time path {}))
  ([path {:keys [nofollow-links]}]
   (get-attribute path "basic:lastModifiedTime" {:nofollow-links nofollow-links})))

(defn size
  [path]
  (try
    (Files/size (as-path path))
    (catch NoSuchFileException _ nil)))

(defn delete
  [path]
  (Files/delete (as-path path)))

(defn delete-dir
  "Delete a directory tree."
  [root]
  (when (directory? root)
    (doseq [path (.listFiles (as-file root))]
      (delete-dir path)))
  (delete root))

(def ^{:doc "A map of file extensions to mime-types."}
  default-mime-types
  {"7z"       "application/x-7z-compressed"
   "aac"      "audio/aac"
   "ai"       "application/postscript"
   "appcache" "text/cache-manifest"
   "asc"      "text/plain"
   "atom"     "application/atom+xml"
   "avi"      "video/x-msvideo"
   "bin"      "application/octet-stream"
   "bmp"      "image/bmp"
   "bz2"      "application/x-bzip"
   "class"    "application/octet-stream"
   "cer"      "application/pkix-cert"
   "crl"      "application/pkix-crl"
   "crt"      "application/x-x509-ca-cert"
   "css"      "text/css"
   "csv"      "text/csv"
   "deb"      "application/x-deb"
   "dart"     "application/dart"
   "dll"      "application/octet-stream"
   "dmg"      "application/octet-stream"
   "dms"      "application/octet-stream"
   "doc"      "application/msword"
   "dvi"      "application/x-dvi"
   "edn"      "application/edn"
   "eot"      "application/vnd.ms-fontobject"
   "eps"      "application/postscript"
   "etx"      "text/x-setext"
   "exe"      "application/octet-stream"
   "flv"      "video/x-flv"
   "flac"     "audio/flac"
   "gif"      "image/gif"
   "gz"       "application/gzip"
   "htm"      "text/html"
   "html"     "text/html"
   "ico"      "image/x-icon"
   "iso"      "application/x-iso9660-image"
   "jar"      "application/java-archive"
   "jpe"      "image/jpeg"
   "jpeg"     "image/jpeg"
   "jpg"      "image/jpeg"
   "js"       "text/javascript"
   "json"     "application/json"
   "lha"      "application/octet-stream"
   "lzh"      "application/octet-stream"
   "mov"      "video/quicktime"
   "m3u8"     "application/x-mpegurl"
   "m4v"      "video/mp4"
   "mjs"      "text/javascript"
   "mp3"      "audio/mpeg"
   "mp4"      "video/mp4"
   "mpd"      "application/dash+xml"
   "mpe"      "video/mpeg"
   "mpeg"     "video/mpeg"
   "mpg"      "video/mpeg"
   "oga"      "audio/ogg"
   "ogg"      "audio/ogg"
   "ogv"      "video/ogg"
   "pbm"      "image/x-portable-bitmap"
   "pdf"      "application/pdf"
   "pgm"      "image/x-portable-graymap"
   "png"      "image/png"
   "pnm"      "image/x-portable-anymap"
   "ppm"      "image/x-portable-pixmap"
   "ppt"      "application/vnd.ms-powerpoint"
   "ps"       "application/postscript"
   "qt"       "video/quicktime"
   "rar"      "application/x-rar-compressed"
   "ras"      "image/x-cmu-raster"
   "rb"       "text/plain"
   "rd"       "text/plain"
   "rss"      "application/rss+xml"
   "rtf"      "application/rtf"
   "sgm"      "text/sgml"
   "sgml"     "text/sgml"
   "svg"      "image/svg+xml"
   "swf"      "application/x-shockwave-flash"
   "tar"      "application/x-tar"
   "tif"      "image/tiff"
   "tiff"     "image/tiff"
   "ts"       "video/mp2t"
   "ttf"      "font/ttf"
   "txt"      "text/plain"
   "webm"     "video/webm"
   "wmv"      "video/x-ms-wmv"
   "woff"     "font/woff"
   "woff2"    "font/woff2"
   "xbm"      "image/x-xbitmap"
   "xls"      "application/vnd.ms-excel"
   "xml"      "text/xml"
   "xpm"      "image/x-xpixmap"
   "xwd"      "image/x-xwindowdump"
   "zip"      "application/zip"})

(defn- filename-ext
  "Returns the file extension of a filename or filepath."
  [filename]
  (when-let [ext (second (re-find #"\.([^./\\]+)$" filename))]
    (clj-str/lower-case ext)))

(defn ext-mime-type
  "Get the mimetype from the filename extension. Takes an optional map of
  extensions to mimetypes that overrides values in the default-mime-types map."
  ([filename]
   (ext-mime-type filename {}))
  ([filename mime-types]
   (let [mime-types (merge default-mime-types mime-types)]
     (mime-types (filename-ext filename)))))

;; ---------------------------- API ----------------------------
(def ^:private workdir (atom nil))

(defn connect
  [^String url ^String access-key ^String secret-key]
  (let [path (clj-str/replace url "local://" "/")
        ;; Maybe the prefix of a url is local:///
        abspath (clj-str/replace path "//" "/")]
    (if (exists? abspath)
      (do
        (reset! workdir abspath)
        ;; Need to be an absolute path, such as local://data/remote-fs/ -> /data/remote-fs
        {:path abspath
         :isfile (regular-file? abspath)
         :isdir (directory? abspath)})
      (throw (ex-info (format "No such file or directory (%s)" abspath) {})))))

(defn make-bucket
  "Creates a bucket with a name. Does nothing if one exists. Returns nil
  "
  [conn ^String name]
  (try
    (let [file (as-file (join-paths (:path conn) name))]
      (if (.mkdir file)
        name
        nil))
    (catch Exception ex
      nil)))

(defn list-buckets
  "returns maps "
  [conn]
  (->> (as-file (:path conn))
       (.listFiles)
       (filter #(.isDirectory %))
       (map (fn [bucket] {"CreationDate" (str (creation-time bucket))
                          "Name" (.getName bucket)}))))

(defn UUID []
  (java.util.UUID/randomUUID))

(defn NOW []
  (t/format "yMMd-HHmm"  (t/local-date-time)))

(defn put-object
  "Uploads a file object to the bucket. 
   Returns a map of bucket name and file name
  "
  ([conn ^String bucket ^String file-name]
   (let [upload-name (str  (NOW) "_" (UUID) "_" file-name)]
     (put-object conn bucket upload-name file-name)
     {:bucket bucket
      :name upload-name}))
  ([conn ^String bucket ^String upload-name ^String source-file-name]
   (if (exists? source-file-name)
     (let [destdir (join-paths (:path conn) bucket)
           destfile (join-paths destdir upload-name)]
       (io/copy (as-file source-file-name) (as-file destfile))
       {:bucket bucket
        :name upload-name})
     (throw (ex-info (format "No such file (%s)" source-file-name) {})))))

(defn get-object
  "Takes connection and a map of [bucket name] keys as returned by (put-object) or explicit arguments 
   returns java.io.BufferedReader.
   Use clojure.java.io/copy to stream the bucket data files, or HTTP responses
  "
  ([conn {:keys [bucket name]}]
   (let [path (join-paths (:path conn) bucket name)]
     (if (exists? path)
       (io/reader path)
       (throw (ex-info (format "No such file (%s)" path) {})))))
  ([conn bucket name]
   (get-object conn {:bucket bucket :name name})))

(defn download-object
  "Download object to a local path."
  [conn bucket name localpath]
  (io/copy (get-object conn bucket name) (io/file localpath)))

(defn- trim-path
  [abspath]
  (clj-str/replace abspath @workdir ""))

(defn- get-bucket
  [abspath]
  (let [bucket-link (trim-path abspath)
        trimmed-bucket-link (clj-str/replace bucket-link #"^/" "")
        first-item (first (clj-str/split trimmed-bucket-link #"/"))]
    first-item))

(defn get-name
  [abspath]
  (let [bucket-link (trim-path abspath)
        trimmed-bucket-link (clj-str/replace bucket-link #"^/" "")
        rest-item (rest (clj-str/split trimmed-bucket-link #"/"))]
    rest-item))

(defn- objectStat->map
  "helper function for datatype conversion"
  [^File file]
  (let [filepath (.toString file)]
    {:bucket (get-bucket filepath)
     :name (get-name filepath)
     :created-time (creation-time filepath)
     :length (size filepath)
     :etag ""
     :content-type (ext-mime-type filepath)
     :encryption-key nil
     :http-headers {}}))

(defn get-object-meta
  "Returns object metadata as clojure hash-map"
  ([conn bucket name]
   (-> (join-paths (:path conn) bucket name)
       objectStat->map
       (assoc :key name)))
  ([conn {:keys [bucket name]}]
   (get-object-meta conn {:bucket bucket :name name})))

(defmacro swallow-exceptions [& body]
  `(try ~@body (catch Exception e#)))

(defn- objectItem->map
  "Helper function for datatye conversion."
  [^File file]
  (let [filepath (.toString file)]
    {:etag ""
     :last-modified (last-modified-time filepath)
     :key (.getName filepath)
     :owner "biominer"
     :size (size filepath)
     :storage-class "Local"
     :user-metadata ""
     :version-id ""}))

(defn list-objects
  ([conn bucket]
   (list-objects conn bucket "" true))
  ([conn bucket filter]
   (list-objects conn bucket filter true))
  ([conn bucket filter recursive]
   (let [bucket-dir (join-paths (:path conn) bucket)
         files (if recursive
                 (file-seq bucket-dir)
                 (.listFiles bucket-dir))
         filtered-files (if filter
                          (filter #(re-matches (re-pattern filter) %) files)
                          files)]
     (map objectItem->map filtered-files))))

(defn remove-bucket!
  "removes the bucket form the storage"
  [conn bucket-name]
  (let [path (join-paths (:path conn) bucket-name)]
    (when (exists? path)
      (delete-dir path))))

(defn exists-object?
  "Whether the object exists in the bucket"
  [conn bucket-name object]
  (try
    (let [path (join-paths (:path conn) bucket-name object)]
      (exists? path))
    (catch Exception ex
      false)))

(defn remove-object!
  [conn bucket object]
  (let [path (join-paths (:path conn) bucket object)]
    (when (and (exists? path) (regular-file? path))
      (delete path))))

(defn get-upload-url
  "returns presigned and named upload url for direct upload from the client.
  "
  [conn bucket name]
  (throw (ex-info "NotImplementedException" {})))

(defn get-download-url
  "returns a temporary download url for this object with 7day expiration
  "
  ([conn bucket name]
   (let [path (join-paths (:path conn) bucket name)]
     (-> (clj-str/replace path workdir "")
         (clj-str/replace #"^/" ""))))
  ([conn bucket name timeout]
   (get-download-url conn bucket name)))

(defn init
  []
  {:connect          connect
   :make-bucket      make-bucket
   :list-buckets     list-buckets
   :exists           exists-object?
   :put-object       put-object
   :get-object       get-object
   :download-object  download-object
   :list-objects     list-objects
   :remove-bucket    remove-bucket!
   :remove-object    remove-object!
   :get-upload-url   get-upload-url
   :get-download-url get-download-url
   :get-object-meta  get-object-meta})
