(ns com.github.wsgtcyx.destiny-matrix.energy
  (:require [clojure.string :as str]))

(defn reduce-number
  "Matches the website behavior:
  If n > 22, reduce once via (n % 10) + floor(n/10), otherwise keep n."
  [n]
  (when-not (integer? n)
    (throw (ex-info "n must be an integer" {:n n})))
  (when (neg? n)
    (throw (ex-info "n must be a non-negative integer" {:n n})))
  (if (> n 22)
    (+ (mod n 10) (quot n 10))
    n))

(defn calculate-year
  "Sum digits of year, then apply reduce-number once."
  [year]
  (when-not (integer? year)
    (throw (ex-info "year must be an integer" {:year year})))
  (when (neg? year)
    (throw (ex-info "year must be a non-negative integer" {:year year})))
  (->> (str year)
       (map (fn [ch] (- (int ch) (int \0))))
       (reduce + 0)
       (reduce-number)))

(defn- leap-year? [year]
  (or (and (zero? (mod year 4)) (not (zero? (mod year 100))))
      (zero? (mod year 400))))

(defn- days-in-month [year month]
  (cond
    (= month 2) (if (leap-year? year) 29 28)
    (contains? #{4 6 9 11} month) 30
    :else 31))

(defn validate-date!
  "Throws if year/month/day is not a valid calendar date."
  [year month day]
  (when-not (integer? year)
    (throw (ex-info "year must be an integer" {:year year})))
  (when-not (integer? month)
    (throw (ex-info "month must be an integer" {:month month})))
  (when-not (integer? day)
    (throw (ex-info "day must be an integer" {:day day})))
  (when (or (< year 1) (> year 9999))
    (throw (ex-info "year must be between 1 and 9999" {:year year})))
  (when (or (< month 1) (> month 12))
    (throw (ex-info "month must be between 1 and 12" {:month month})))
  (let [max-day (days-in-month year month)]
    (when (or (< day 1) (> day max-day))
      (throw (ex-info "day is out of range for the given month/year"
                      {:year year :month month :day day :max-day max-day}))))
  true)

(defn parse-iso-date
  "Parses YYYY-MM-DD into [year month day]."
  [s]
  (when-not (string? s)
    (throw (ex-info "date must be a string in YYYY-MM-DD format" {:date s})))
  (let [[y m d] (str/split s #"-")]
    (when-not (and (= 4 (count y)) (= 2 (count m)) (= 2 (count d))
                   (re-matches #"\d{4}" y)
                   (re-matches #"\d{2}" m)
                   (re-matches #"\d{2}" d))
      (throw (ex-info "date must be in YYYY-MM-DD format" {:date s})))
    [(parse-long y) (parse-long m) (parse-long d)]))

(defn base-from-ymd
  "Returns [apoint bpoint cpoint] for a birth date.
  By default validates calendar dates; pass {:validate false} to skip."
  ([year month day] (base-from-ymd year month day {}))
  ([year month day {:keys [validate] :or {validate true}}]
   (when validate
     (validate-date! year month day))
   (let [apoint (reduce-number day)
         bpoint month
         cpoint (calculate-year year)]
     [apoint bpoint cpoint])))

(defn calculate-from-base
  "Calculates full Destiny Matrix output from base points apoint/bpoint/cpoint."
  [apoint bpoint cpoint]
  (doseq [[k v] [[:apoint apoint] [:bpoint bpoint] [:cpoint cpoint]]]
    (when-not (integer? v)
      (throw (ex-info (str (name k) " must be an integer") {k v}))))
  (let [r reduce-number
        dpoint (r (+ apoint bpoint cpoint))
        epoint (r (+ apoint bpoint cpoint dpoint))
        fpoint (r (+ apoint bpoint))
        gpoint (r (+ bpoint cpoint))
        hpoint (r (+ dpoint apoint))
        ipoint (r (+ cpoint dpoint))
        jpoint (r (+ dpoint epoint))

        npoint (r (+ cpoint epoint))
        lpoint (r (+ jpoint npoint))
        mpoint (r (+ lpoint npoint))
        kpoint (r (+ jpoint lpoint))

        qpoint (r (+ npoint cpoint))
        rpoint (r (+ jpoint dpoint))
        spoint (r (+ apoint epoint))
        tpoint (r (+ bpoint epoint))

        opoint (r (+ apoint spoint))
        ppoint (r (+ bpoint tpoint))

        upoint (r (+ fpoint gpoint hpoint ipoint))
        vpoint (r (+ epoint upoint))
        wpoint (r (+ spoint epoint))
        xpoint (r (+ tpoint epoint))

        f2point (r (+ fpoint upoint))
        f1point (r (+ fpoint f2point))
        g2point (r (+ gpoint upoint))
        g1point (r (+ gpoint g2point))
        i2point (r (+ ipoint upoint))
        i1point (r (+ ipoint i2point))
        h2point (r (+ hpoint upoint))
        h1point (r (+ hpoint h2point))

        afpoint (r (+ apoint fpoint))
        af1point (r (+ apoint afpoint))
        af2point (r (+ apoint af1point))
        af3point (r (+ afpoint af1point))
        af4point (r (+ afpoint fpoint))
        af5point (r (+ afpoint af4point))
        af6point (r (+ af4point fpoint))
        fbpoint (r (+ fpoint bpoint))
        fb1point (r (+ fpoint fbpoint))
        fb2point (r (+ fpoint fb1point))
        fb3point (r (+ fbpoint fb1point))
        fb4point (r (+ fbpoint bpoint))
        fb5point (r (+ fbpoint fb4point))
        fb6point (r (+ fb4point bpoint))
        bgpoint (r (+ bpoint gpoint))
        bg1point (r (+ bpoint bgpoint))
        bg2point (r (+ bpoint bg1point))
        bg3point (r (+ bgpoint bg1point))
        bg4point (r (+ bgpoint gpoint))
        bg5point (r (+ bgpoint bg4point))
        bg6point (r (+ bg4point gpoint))
        gcpoint (r (+ gpoint cpoint))
        gc1point (r (+ gpoint gcpoint))
        gc2point (r (+ gpoint gc1point))
        gc3point (r (+ gcpoint gc1point))
        gc4point (r (+ gcpoint cpoint))
        gc5point (r (+ gcpoint gc4point))
        gc6point (r (+ gc4point cpoint))
        cipoint (r (+ cpoint ipoint))
        ci1point (r (+ cpoint cipoint))
        ci2point (r (+ cpoint ci1point))
        ci3point (r (+ cipoint ci1point))
        ci4point (r (+ cipoint ipoint))
        ci5point (r (+ cipoint ci4point))
        ci6point (r (+ ci4point ipoint))
        idpoint (r (+ ipoint dpoint))
        id1point (r (+ ipoint idpoint))
        id2point (r (+ ipoint id1point))
        id3point (r (+ idpoint id1point))
        id4point (r (+ idpoint dpoint))
        id5point (r (+ idpoint id4point))
        id6point (r (+ id4point dpoint))
        dhpoint (r (+ dpoint hpoint))
        dh1point (r (+ dpoint dhpoint))
        dh2point (r (+ dpoint dh1point))
        dh3point (r (+ dhpoint dh1point))
        dh4point (r (+ dhpoint hpoint))
        dh5point (r (+ dhpoint dh4point))
        dh6point (r (+ dh4point hpoint))
        hapoint (r (+ hpoint apoint))
        ha1point (r (+ hpoint hapoint))
        ha2point (r (+ hpoint ha1point))
        ha3point (r (+ hapoint ha1point))
        ha4point (r (+ hapoint apoint))
        ha5point (r (+ hapoint ha4point))
        ha6point (r (+ ha4point apoint))

        skypoint (r (+ bpoint dpoint))
        earthpoint (r (+ apoint cpoint))
        perspurpose (r (+ skypoint earthpoint))
        femalepoint (r (+ gpoint hpoint))
        malepoint (r (+ fpoint ipoint))
        socialpurpose (r (+ femalepoint malepoint))
        generalpurpose (r (+ perspurpose socialpurpose))
        planetarypurpose (r (+ socialpurpose generalpurpose))

        chart-heart {:sahphysics apoint
                     :ajphysics opoint
                     :vishphysics spoint
                     :anahphysics wpoint
                     :manphysics epoint
                     :svadphysics jpoint
                     :mulphysics cpoint

                     :sahenergy bpoint
                     :ajenergy ppoint
                     :vishenergy tpoint
                     :anahenergy xpoint
                     :manenergy epoint
                     :svadenergy npoint
                     :mulenergy dpoint

                     :sahemotions (r (+ apoint bpoint))
                     :ajemotions (r (+ opoint ppoint))
                     :vishemotions (r (+ spoint tpoint))
                     :anahemotions (r (+ wpoint xpoint))
                     :manemotions (r (+ epoint epoint))
                     :svademotions (r (+ jpoint npoint))
                     :mulemotions (r (+ cpoint dpoint))}]
    {:base {:apoint apoint :bpoint bpoint :cpoint cpoint}
     :points {:apoint apoint
              :bpoint bpoint
              :cpoint cpoint
              :dpoint dpoint
              :epoint epoint
              :fpoint fpoint
              :gpoint gpoint
              :hpoint hpoint
              :ipoint ipoint
              :jpoint jpoint
              :kpoint kpoint
              :lpoint lpoint
              :mpoint mpoint
              :npoint npoint
              :opoint opoint
              :ppoint ppoint
              :qpoint qpoint
              :rpoint rpoint
              :spoint spoint
              :tpoint tpoint
              :upoint upoint
              :vpoint vpoint
              :wpoint wpoint
              :xpoint xpoint
              :f2point f2point
              :f1point f1point
              :g2point g2point
              :g1point g1point
              :i2point i2point
              :i1point i1point
              :h2point h2point
              :h1point h1point}
     :purposes {:skypoint skypoint
                :earthpoint earthpoint
                :perspurpose perspurpose
                :femalepoint femalepoint
                :malepoint malepoint
                :socialpurpose socialpurpose
                :generalpurpose generalpurpose
                :planetarypurpose planetarypurpose}
     :chart-heart (assoc chart-heart
                         :resultphysics (r (+ (:sahphysics chart-heart)
                                             (:ajphysics chart-heart)
                                             (:vishphysics chart-heart)
                                             (:anahphysics chart-heart)
                                             (:manphysics chart-heart)
                                             (:svadphysics chart-heart)
                                             (:mulphysics chart-heart)))
                         :resultenergy (r (+ (:sahenergy chart-heart)
                                            (:ajenergy chart-heart)
                                            (:vishenergy chart-heart)
                                            (:anahenergy chart-heart)
                                            (:manenergy chart-heart)
                                            (:svadenergy chart-heart)
                                            (:mulenergy chart-heart)))
                         :resultemotions (r (+ (:sahemotions chart-heart)
                                              (:ajemotions chart-heart)
                                              (:vishemotions chart-heart)
                                              (:anahemotions chart-heart)
                                              (:manemotions chart-heart)
                                              (:svademotions chart-heart)
                                              (:mulemotions chart-heart))))
     :years {:afpoint afpoint
             :af1point af1point
             :af2point af2point
             :af3point af3point
             :af4point af4point
             :af5point af5point
             :af6point af6point
             :fbpoint fbpoint
             :fb1point fb1point
             :fb2point fb2point
             :fb3point fb3point
             :fb4point fb4point
             :fb5point fb5point
             :fb6point fb6point
             :bgpoint bgpoint
             :bg1point bg1point
             :bg2point bg2point
             :bg3point bg3point
             :bg4point bg4point
             :bg5point bg5point
             :bg6point bg6point
             :gcpoint gcpoint
             :gc1point gc1point
             :gc2point gc2point
             :gc3point gc3point
             :gc4point gc4point
             :gc5point gc5point
             :gc6point gc6point
             :cipoint cipoint
             :ci1point ci1point
             :ci2point ci2point
             :ci3point ci3point
             :ci4point ci4point
             :ci5point ci5point
             :ci6point ci6point
             :idpoint idpoint
             :id1point id1point
             :id2point id2point
             :id3point id3point
             :id4point id4point
             :id5point id5point
             :id6point id6point
             :dhpoint dhpoint
             :dh1point dh1point
             :dh2point dh2point
             :dh3point dh3point
             :dh4point dh4point
             :dh5point dh5point
             :dh6point dh6point
             :hapoint hapoint
             :ha1point ha1point
             :ha2point ha2point
             :ha3point ha3point
             :ha4point ha4point
             :ha5point ha5point
             :ha6point ha6point}}))

(defn calculate
  "High-level entry:
  - (calculate \"YYYY-MM-DD\" {:validate true})
  - (calculate {:year 1990 :month 12 :day 31} {:validate true})"
  ([input] (calculate input {}))
  ([input {:keys [validate] :or {validate true} :as opts}]
   (cond
     (string? input)
     (let [[year month day] (parse-iso-date input)
           [apoint bpoint cpoint] (base-from-ymd year month day {:validate validate})]
       (calculate-from-base apoint bpoint cpoint))

     (map? input)
     (let [year (or (:year input) (:y input))
           month (or (:month input) (:m input))
           day (or (:day input) (:d input))
           [apoint bpoint cpoint] (base-from-ymd year month day {:validate validate})]
       (calculate-from-base apoint bpoint cpoint))

     :else
     (throw (ex-info "unsupported input; use a YYYY-MM-DD string or {:year :month :day} map"
                     {:input input :opts opts})))))
