(ns reveal.actions
  (:require [clojure.core.protocols :as p]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]))

(defonce ^:private *registry
  (atom {::actions (sorted-map-by (fn [[^String label-a id-a] [^String label-b id-b]]
                                    (let [n ( .compareToIgnoreCase label-a label-b)]
                                      (if (zero? n)
                                        (compare id-a id-b)
                                        n))))
         ::keys {}}))

(defn- unregister [registry id]
  (let [key (get-in registry [::keys id] ::not-found)]
    (cond-> registry
            (not= key ::not-found)
            (-> (update ::keys dissoc id)
                (update ::actions dissoc key)))))

(defn- register [registry action]
  (let [{:keys [id priority label]
         :or {priority 0}} action
        key [label id]]
    (-> registry
        (unregister id)
        (assoc-in [::keys id] key)
        (assoc-in [::actions key] action))))

(s/def ::id any?)
(s/def ::label string?)
(s/def ::check fn?)
(s/def ::action
  (s/keys :req-un [::id ::check ::label]))

(defn reg! [action]
  (s/assert ::action action)
  (swap! *registry register action)
  (:id action))

(reg! {:id ::nav
       :label "Navigate"
       :check (fn [values]
                (when (<= 2 (count values))
                  (let [[x ann] (peek values)
                        coll (p/datafy (first (peek (pop values))))
                        c (class coll)]
                    (when (or (instance? (:on-interface p/Navigable) coll)
                              (contains? (meta coll) `p/nav)
                              (seq (set/intersection (disj (supers c) Object)
                                                     (set (keys (:impls p/Navigable))))))
                      (let [{:reveal.nav/keys [key val index]
                             :or {key ::not-found
                                  val ::not-found}} ann]
                        (cond
                          index #(p/nav coll index x)
                          (not= key ::not-found) #(p/nav coll key x)
                          (not= val ::not-found) #(p/nav coll x val)))))))})

(reg! {:id ::show-exception
       :label "Show Exception"
       :check (fn [[[prepl-output]]]
                (when (:exception prepl-output)
                  #(:val prepl-output)))})

(reg! {:id ::value
       :label "Show value"
       :check (fn [values+metas]
                (when-let [v+m (peek values+metas)]
                  #(first v+m)))})

(reg! {:id ::rand
       ;:label "Generate random number, in fact, the number so random your butt explodes"
       :label "Generate random number"
       :check (constantly rand)})

(defn- check [action values]
  (try
    (let [f ((:check action) values)]
      (when (ifn? f)
        (assoc action :invoke f)))
    (catch Exception _)))

(defn collect [values]
  (let [registry @*registry]
    (into []
          (keep #(check % values))
          (vals (::actions registry)))))