(ns reveal.parse
  (:require [reveal.stream :as stream]
            [reveal.style :as style]
            [clojure.main :as m])
  (:import [clojure.lang Keyword Symbol IPersistentMap IPersistentVector IPersistentSet Fn IPersistentList ISeq MultiFn
                         IRef Var Volatile Namespace IRecord Delay IBlockingDeref]
           [java.util.regex Pattern]
           [java.io File]))

(defn- identity-hash-code [x]
  (let [hash (System/identityHashCode x)]
    (stream/as hash
      (stream/raw-string (format "0x%x" hash) {:fill [:palette :scalar]}))))

(extend-protocol stream/Parse
  Object
  (parse [x]
    (stream/multiline (pr-str x) {:fill [:palette :object]}))
  nil
  (parse [x]
    (stream/raw-string (pr-str x) {:fill [:palette :scalar]}))
  Boolean
  (parse [x]
    (stream/raw-string (pr-str x) {:fill [:palette :scalar]}))
  String
  (parse [s]
    (stream/raw-string (pr-str s) {:fill [:palette :string]}))
  Character
  (parse [ch]
    (stream/raw-string (pr-str ch) {:fill [:palette :string]}))
  Keyword
  (parse [k]
    (stream/ensure-simple-string (str k) {:fill [:palette :keyword]}))
  Symbol
  (parse [sym]
    (stream/ensure-simple-string (str sym) {:fill [:palette :symbol]}))
  Number
  (parse [n]
    (stream/ensure-simple-string (str n) {:fill [:palette :scalar]}))
  Float
  (parse [n]
    (stream/raw-string
      (cond
        (= Float/POSITIVE_INFINITY n) "##Inf"
        (= Float/NEGATIVE_INFINITY n) "##-Inf"
        (Float/isNaN n) "##NaN"
        :else (str n))
      {:fill [:palette :scalar]}))
  Double
  (parse [n]
    (stream/raw-string
      (cond
        (= Double/POSITIVE_INFINITY n) "##Inf"
        (= Double/NEGATIVE_INFINITY n) "##-Inf"
        (Double/isNaN n) "##NaN"
        :else (str n))
      {:fill [:palette :scalar]}))
  IPersistentMap
  (parse [m]
    (stream/horizontal
      (stream/raw-string "{" {:fill [:palette :object]})
      (stream/vertical (stream/entries m))
      (stream/raw-string "}" {:fill [:palette :object]})))
  IRecord
  (parse [rec]
    (stream/horizontal
      (stream/raw-string (str "#" (.getName (class rec)) "{") {:fill [:palette :object]})
      (stream/vertical (stream/entries rec))
      (stream/raw-string "}" {:fill [:palette :object]})))
  IPersistentVector
  (parse [v]
    (stream/horizontal
      (stream/raw-string "[" {:fill [:palette :object]})
      (stream/items v)
      (stream/raw-string "]" {:fill [:palette :object]})))
  IPersistentList
  (parse [v]
    (stream/horizontal
      (stream/raw-string "(" {:fill [:palette :object]})
      (stream/items v)
      (stream/raw-string ")" {:fill [:palette :object]})))
  ISeq
  (parse [xs]
    (stream/horizontal
      (stream/raw-string "(" {:fill [:palette :object]})
      (stream/sequential xs)
      (stream/raw-string ")" {:fill [:palette :object]})))
  IPersistentSet
  (parse [s]
    (stream/horizontal
      (stream/raw-string "#{" {:fill [:palette :object]})
      (stream/items s)
      (stream/raw-string "}" {:fill [:palette :object]})))
  Throwable
  (parse [t]
    (stream/horizontal
      (stream/raw-string "#error" {:fill [:palette :object]})
      (stream/stream (Throwable->map t))))
  MultiFn
  (parse [f]
    (stream/horizontal
      (stream/raw-string "#multi-fn[" {:fill [:palette :object]})
      (identity-hash-code f)
      (stream/raw-string " ")
      (stream/stream (.-dispatchFn f))
      (stream/raw-string "]" {:fill [:palette :object]})))
  Fn
  (parse [f]
    (stream/ensure-simple-string
      (Compiler/demunge (.getName (class f)))
      {:fill [:palette :object]}))
  Pattern
  (parse [re]
    (stream/horizontal
      (stream/raw-string "#" {:fill [:palette :object]})
      (stream/ensure-simple-string (str \" re \") {:fill [:palette :string]})))
  Var
  (parse [var]
    (stream/raw-string (pr-str var) {:fill [:palette :object]}))
  Namespace
  (parse [ns]
    (stream/raw-string (name (.getName ns)) {:fill [:palette :object]}))
  Class
  (parse [class]
    (stream/raw-string (.getName class) {:fill [:palette :object]}))
  IRef
  (parse [*ref]
    (stream/horizontal
      (stream/raw-string (str "#" (.toLowerCase (.getSimpleName (class *ref))) "[") {:fill [:palette :object]})
      (identity-hash-code *ref)
      (stream/raw-string " ")
      (stream/stream @*ref)
      (stream/raw-string "]" {:fill [:palette :object]})))
  File
  (parse [file]
    (stream/horizontal
      (stream/raw-string "#file[" {:fill [:palette :object]})
      stream/separator
      (stream/ensure-simple-string (str file) {:fill [:palette :string]})
      stream/separator
      (stream/raw-string "]" {:fill [:palette :object]})))
  Delay
  (parse [*delay]
    (stream/horizontal
      (stream/raw-string "#delay[" {:fill [:palette :object]})
      (identity-hash-code *delay)
      (stream/raw-string " ")
      (if (realized? *delay)
        (stream/stream @*delay)
        (stream/raw-string "..." {:fill [:palette :util]}))
      (stream/raw-string "]" {:fill [:palette :object]})))
  IBlockingDeref
  (parse [*blocking-deref]
    (let [class-name (.getName (class *blocking-deref))]
      (cond
        (.startsWith class-name "clojure.core$promise$reify")
        (stream/horizontal
          (stream/raw-string "#promise[" {:fill [:palette :object]})
          (identity-hash-code *blocking-deref)
          (stream/raw-string " ")
          (if (realized? *blocking-deref)
            (stream/stream @*blocking-deref)
            (stream/raw-string "..." {:fill [:palette :util]}))
          (stream/raw-string "]" {:fill [:palette :object]}))

        (.startsWith class-name "clojure.core$future_call$reify")
        (stream/horizontal
          (stream/raw-string "#future[" {:fill [:palette :object]})
          (identity-hash-code *blocking-deref)
          (stream/raw-string " ")
          (if (realized? *blocking-deref)
            (stream/stream @*blocking-deref)
            (stream/raw-string "..." {:fill [:palette :util]}))
          (stream/raw-string "]" {:fill [:palette :object]}))

        :else
        (stream/multiline (pr-str *blocking-deref) {:fill [:palette :object]}))))
  Volatile
  (parse [*ref]
    (stream/horizontal
      (stream/raw-string "#volatile[" {:fill [:palette :object]})
      (identity-hash-code *ref)
      (stream/raw-string " ")
      (stream/stream @*ref)
      (stream/raw-string "]" {:fill [:palette :object]}))))

(defn prepl-output [x]
  (stream/as x
    (if (:exception x)
      (cond-> (stream/multiline (-> x :val m/ex-triage m/ex-str) {:fill [:palette :error]})
              (:form x)
              (as-> err-output
                    (stream/vertical
                      (stream/multiline (:form x) {:fill [:palette :util]})
                      err-output)))
      (case (:tag x)
        :ret (stream/vertical
               (stream/multiline (:form x) {:fill [:palette :util]})
               (stream/horizontal
                 (stream/raw-string "=>" {:fill [:palette :util]})
                 stream/separator
                 (stream/raw-string " " {:fill [:palette :util]})
                 (stream/stream (:val x))))
        :out (stream/multiline (:val x) {:fill [:palette :string]})
        :err (stream/multiline (:val x) {:fill [:palette :error]})
        :tap (stream/horizontal
               (stream/raw-string "tap>" {:fill [:palette :util]})
               stream/separator
               (stream/raw-string " " {:fill [:palette :util]})
               (stream/stream (:val x)))
        (stream/parse x)))))

(defn action-output [action ret]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill [:palette :util]})
        (stream/stream (:id action)))
      (stream/horizontal
        (stream/raw-string "$> " {:fill [:palette :util]})
        (stream/stream ret)))))

(defn action-error-output [action ex]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill [:palette :util]})
        (stream/stream (:id action)))
      (stream/stream ex nil))))

(defn system-out [line]
  (stream/as line
    (stream/raw-string line {:fill [:palette :string]})))

(defn system-err [line]
  (stream/as line
    (stream/raw-string line {:fill [:palette :error]})))