(ns reveal.core
  (:require [clojure.core.server :as server]
            [cljfx.api :as fx]
            [clojure.edn :as edn]
            [reveal.font :as font]
            [clojure.main :as m]
            [reveal.stream :as stream]
            [reveal.output-panel :as output-panel]
            [reveal.event :as event]
            [reveal.writer-output-stream :as writer-output-stream]
            [reveal.parse :as parse]
            [reveal.style :as style]
            [reveal.layout :as layout])
  (:import [javafx.event Event]
           [java.util.concurrent Executors ThreadFactory ArrayBlockingQueue]
           [java.io PrintStream]))

(defn- root-view-impl [{:keys [showing canvas reveal/css]}]
  {:fx/type :stage
   :on-close-request #(.consume ^Event %)
   :showing showing
   :width 400
   :height 500
   :on-focused-changed {::event/type ::output-panel/on-window-focus-changed
                        :path [:canvas]}
   :scene {:fx/type :scene
           :stylesheets [(style/url)]
           :root {:fx/type :v-box
                  :style {:-fx-background-color (css [:palette :background])}
                  :children [#_{:fx/type :list-view
                                :focus-traversable false
                                :items ["blep" "nav" "meta" "doc"]}
                             {:fx/type output-panel/view
                              :v-box/vgrow :always
                              :path [:canvas]
                              :layout canvas}]}}})

(defn- root-view [props]
  {:fx/type fx/ext-set-env
   :env {:reveal/css style/dark}
   :desc {:fx/type fx/ext-get-env
          :env [:reveal/css]
          :desc (assoc props :fx/type root-view-impl)}})

(defn oneduce
  ([xf x]
   (oneduce xf identity x))
  ([xf f x]
   (let [rf (xf (completing #(f %2)))]
     (rf (rf nil x)))))

(defn- execute-effect [action output!]
  (future
    (try
      (let [ret ((:invoke action))]
        (output! (parse/action-output action ret)))
      (catch Exception e
        (output! (parse/action-error-output action e))))))

(def ^:private daemon-executor
  (let [*counter (atom 0)
        factory (reify ThreadFactory
                  (newThread [_ runnable]
                    (doto (Thread. runnable (str "reveal-agent-pool-" (swap! *counter inc)))
                      (.setDaemon true))))]
    (Executors/newCachedThreadPool factory)))

(defn- line-print-stream [line-fn]
  (let [sb (StringBuilder.)]
    (-> #(doseq [^char ch %]
           (if (= \newline ch)
             (let [str (.toString sb)]
               (.delete sb 0 (.length sb))
               (line-fn str))
             (.append sb ch)))
        (PrintWriter-on nil)
        (writer-output-stream/make)
        (PrintStream. true "UTF-8"))))

(defn- wrap-err-out [f output!]
  (fn []
    (let [out System/out
          err System/err]
      (System/setOut (line-print-stream #(-> % parse/system-out output!)))
      (System/setErr (line-print-stream #(-> % parse/system-err output!)))
      (f)
      (System/setOut out)
      (System/setErr err))))

(defn- wrap-renderer [f *state event-handler]
  (fn []
    (let [renderer (fx/create-renderer
                     :opts {:fx.opt/map-event-handler event-handler}
                     :middleware (fx/wrap-map-desc #(root-view %)))]
      (fx/mount-renderer *state renderer)
      (f)
      (fx/unmount-renderer *state renderer))))

(defn- wrap-output-streaming [f out-queue xform to-fn]
  (fn []
    (let [output-thread (doto (Thread.
                                ^Runnable
                                (fn []
                                  (while (not (.isInterrupted (Thread/currentThread)))
                                    (try
                                      (let [x (.take out-queue)
                                            x (if (= ::stream-nil x) nil x)]
                                        (oneduce xform to-fn x))
                                      (catch InterruptedException _))))
                                "reveal-stream-thread")
                          (.setDaemon true))]
      (.start output-thread)
      (f)
      (.interrupt output-thread))))


(defn start [& {:keys [prepl streams]
                :or {prepl server/prepl
                     streams false}}]
  (let [out *out*
        err *err*
        font (font/make "Fantasque Sans Mono" 15)
        output-xf (comp (stream/stream-xf font)
                        (partition-all 128))
        out-queue (ArrayBlockingQueue. 1024)
        output! #(.put out-queue (if (nil? %) ::stream-nil %))
        *state (atom {:canvas (layout/make {:font font
                                            :lines []
                                            :canvas-width 0.0
                                            :canvas-height 0.0})
                      :showing true})
        _ (def -state *state)
        event-handler (-> event/handle
                          (fx/wrap-co-effects
                            {:state (fx/make-deref-co-effect *state)})
                          (fx/wrap-effects
                            {:dispatch fx/dispatch-effect
                             :state (fx/make-reset-effect *state)
                             :execute (fn [action _]
                                        (execute-effect action output!))})
                          (fx/wrap-async :error-handler #(binding [*out* err]
                                                           (prn %2))
                                         :fx/executor daemon-executor))
        repl (-> #(prepl *in* (fn [x]
                                (output! (parse/prepl-output x))
                                (binding [*out* out
                                          *print-length* 5]
                                  (if (:exception x)
                                    (binding [*out* err]
                                      (println (m/ex-str (m/ex-triage (:val x))))
                                      (flush))
                                    (do
                                      (case (:tag x)
                                        :out (println (:val x))
                                        :err (binding [*out* err]
                                               (println (:val x)))
                                        (:tap :ret) (prn (:val x))
                                        (prn x))
                                      (flush)))
                                  (print (str (:ns x) "=> "))
                                  (flush))))
                 (wrap-output-streaming
                   out-queue
                   output-xf
                   #(event-handler {::event/type ::output-panel/add-lines
                                    :path [:canvas]
                                    :fx/event %}))
                 (wrap-renderer *state event-handler)
                 (cond-> streams (wrap-err-out output!)))]
    (repl)
    nil))

#_(-> -state deref :canvas keys)

#_(start)

#_(/ 1 0)

#_(do
    (require 'criterium.core)
    (criterium.core/quick-bench (+ 1 2)))

#_(range 10000)

#_(do :repl/quit)

#_(.printStackTrace (RuntimeException.))

(defn -main [& args]
  (apply start (map edn/read-string args)))

#_(stream/as 1 (stream/raw-string "aaa\tb" {:fill "#ccc"}))

#_(font/char-width (font/make "monospace" 14) :regular \a)

#_(fx/on-fx-thread
    (fx/create-component
      {:fx/type :stage
       :showing true
       :scene {:fx/type :scene
               :root {:fx/type :v-box
                      :padding 50
                      :children [{:fx/type :button
                                  :on-action (fn [e]
                                               (let [target ^javafx.scene.Node (.getTarget e)
                                                     pos (.localToScreen target 0.0 0.0)]
                                                 (.show
                                                   ^javafx.stage.Popup
                                                   (fx/instance
                                                     (fx/create-component
                                                       {:fx/type :popup
                                                        :anchor-location :content-bottom-left
                                                        :content [{:fx/type :v-box
                                                                   :padding 40
                                                                   :style {:-fx-background-color :red}
                                                                   :children [{:fx/type :button
                                                                               :text "woop"}]}]}))
                                                   (.getWindow (.getScene target))
                                                   (.getX pos)
                                                   (.getY pos))))
                                  :text "Click me"}]}}}))