(ns reveal.core
  (:require [clojure.core.server :as server]
            [cljfx.api :as fx]
            [reveal.font :as font]
            [clojure.main :as m]
            [reveal.stream :as stream]
            [reveal.canvas :as canvas]
            [reveal.event :as event]
            [reveal.parse :as parse])
  (:import [javafx.event Event]
           [java.util.concurrent Executors ThreadFactory ArrayBlockingQueue]))

(defn- root-view [{:keys [showing canvas]}]
  {:fx/type :stage
   :on-close-request #(.consume ^Event %)
   :showing showing
   :width 400
   :height 500
   :on-focused-changed {::event/type ::canvas/on-window-focus-changed}
   :scene {:fx/type :scene
           :root {:fx/type :v-box
                  :style {:-fx-background-color "#4a4a4a"}
                  :children [{:fx/type canvas/view
                              :v-box/vgrow :always
                              :layout canvas}]}}})

(defn oneduce
  ([xf x]
   (oneduce xf identity x))
  ([xf f x]
   (let [rf (xf (completing #(f %2)))]
     (rf (rf nil x)))))

(defn- execute-effect [action stream!]
  (future
    (try
      (let [ret ((:invoke action))]
        (stream! (parse/action-output action ret)))
      (catch Exception e
        (stream! (parse/action-error-output action e))))))

(def ^:private daemon-executor
  (let [*counter (atom 0)
        factory (reify ThreadFactory
                  (newThread [_ runnable]
                    (doto (Thread. runnable (str "reveal-agent-pool-" (swap! *counter inc)))
                      (.setDaemon true))))]
    (Executors/newCachedThreadPool factory)))

(defn start [& {:keys [prepl]
                :or {prepl server/prepl}}]
  (let [out *out*
        err *err*
        font (font/make {:family "Fantasque Sans Mono" :size 15})
        stream-xf (comp (stream/stream-xf font)
                        (partition-all 128))
        out-queue (ArrayBlockingQueue. 1024)
        stream! #(.put out-queue (if (nil? %) ::stream-nil %))
        *state (atom {:canvas (canvas/layout {:font font
                                              :lines []
                                              :canvas-width 0.0
                                              :canvas-height 0.0
                                              :scroll-x 0.0
                                              :scroll-y 0.0})
                      :showing true})
        event-handler (-> event/handle
                          (fx/wrap-co-effects
                            {:state (fx/make-deref-co-effect *state)})
                          (fx/wrap-effects
                            {:dispatch fx/dispatch-effect
                             :state (fx/make-reset-effect *state)
                             :execute (fn [action _]
                                        (execute-effect action stream!))})
                          (fx/wrap-async :error-handler #(binding [*out* err]
                                                           (prn %2))
                                         :fx/executor daemon-executor))
        stream-thread (doto (Thread.
                              ^Runnable
                              (fn []
                                (while (not (.isInterrupted (Thread/currentThread)))
                                  (try
                                    (let [x (.take out-queue)
                                          x (if (= ::stream-nil x) nil x)]
                                      (oneduce stream-xf
                                               #(event-handler {::event/type ::canvas/add-lines :fx/event %})
                                               x))
                                    (catch InterruptedException _))))
                              "reveal-stream-thread")
                        (.setDaemon true)
                        (.start))
        renderer (fx/create-renderer
                   :opts {:fx.opt/map-event-handler event-handler}
                   :middleware (fx/wrap-map-desc #(root-view %)))]
    (fx/mount-renderer *state renderer)
    (prepl *in* (fn [x]
                  (stream! (parse/prepl-output x))
                  (binding [*out* out
                            *print-length* 5]
                    (if (:exception x)
                      (binding [*out* err]
                        (println (m/ex-str (m/ex-triage (:val x))))
                        (flush))
                      (do
                        (case (:tag x)
                          :out (println (:val x))
                          :err (binding [*out* err]
                                 (println (:val x)))
                          (:tap :ret) (prn (:val x))
                          (prn x))
                        (flush)))
                    (print (str (:ns x) "=> "))
                    (flush))))
    (.interrupt stream-thread)
    (fx/unmount-renderer *state renderer)
    nil))

#_(start)

#_(/ 1 0)

#_(range 10000)

#_(do :repl/quit)

(defn -main [] (start))
