(ns reveal.parse
  (:require [reveal.stream :as stream]
            [clojure.main :as m])
  (:import [clojure.lang Keyword Symbol IPersistentMap IPersistentVector IPersistentSet Fn IPersistentList ISeq MultiFn IRef Var IFn Volatile]
           [java.util.regex Pattern]))

(defn- identity-hash-code [x]
  (let [hash (System/identityHashCode x)]
    (stream/as hash
      (stream/raw-string (format "0x%x" hash) {:fill "#aaf"}))))

(extend-protocol stream/Parse
  Object
  (parse [x]
    (stream/multiline (pr-str x) {:fill "#ccc"}))
  nil
  (parse [x]
    (stream/raw-string (pr-str x) {:fill "#fae"}))
  Boolean
  (parse [x]
    (stream/raw-string (pr-str x) {:fill "#fae"}))
  String
  (parse [s]
    (stream/raw-string (pr-str s) {:fill "#cec"}))
  Character
  (parse [ch]
    (stream/raw-string (pr-str ch) {:fill "#cec"}))
  Keyword
  (parse [k]
    (if-let [ns (namespace k)]
      (stream/ensure-simple-string
        ":" {:fill "#c7e"}
        ns {:fill "#c9e"}
        (str "/" (name k)) {:fill "#c7e"})
      (stream/ensure-simple-string (str k) {:fill "#c7e"})))
  Symbol
  (parse [sym]
    (if-let [ns (namespace sym)]
      (stream/ensure-simple-string
        ns {:fill "#ace"}
        (str "/" (name sym)) {:fill "#8ce"})
      (stream/ensure-simple-string (str sym) {:fill "#8ce"})))
  Number
  (parse [n]
    (stream/ensure-simple-string (str n) {:fill "#aaf"}))
  Float
  (parse [n]
    (stream/raw-string
      (cond
        (= Float/POSITIVE_INFINITY n) "##Inf"
        (= Float/NEGATIVE_INFINITY n) "##-Inf"
        (Float/isNaN n) "##NaN"
        :else (str n))
      {:fill "#aaf"}))
  Double
  (parse [n]
    (stream/raw-string
      (cond
        (= Double/POSITIVE_INFINITY n) "##Inf"
        (= Double/NEGATIVE_INFINITY n) "##-Inf"
        (Double/isNaN n) "##NaN"
        :else (str n))
      {:fill "#aaf"}))
  IPersistentMap
  (parse [m]
    (stream/horizontal
      (stream/raw-string "{" {:fill "#ccf"})
      (stream/vertical (stream/entries m))
      (stream/raw-string "}" {:fill "#ccf"})))
  IPersistentVector
  (parse [v]
    (stream/horizontal
      (stream/raw-string "[" {:fill "#ccf"})
      (stream/items v)
      (stream/raw-string "]" {:fill "#ccf"})))
  IPersistentList
  (parse [v]
    (stream/horizontal
      (stream/raw-string "(" {:fill "#ccf"})
      (stream/items v)
      (stream/raw-string ")" {:fill "#ccf"})))
  ISeq
  (parse [xs]
    (stream/horizontal
      (stream/raw-string "(" {:fill "#ccf"})
      (stream/sequential xs)
      (stream/raw-string ")" {:fill "#ccf"})))
  IPersistentSet
  (parse [s]
    (stream/horizontal
      (stream/raw-string "#{" {:fill "#ccf"})
      (stream/items s)
      (stream/raw-string "}" {:fill "#ccf"})))
  Throwable
  (parse [t]
    (stream/horizontal
      (stream/raw-string "#error" {:fill "#ccf"})
      (stream/stream (Throwable->map t))))
  MultiFn
  (parse [f]
    (stream/horizontal
      (stream/raw-string "#multi-fn[" {:fill "#ccf"})
      (identity-hash-code f)
      (stream/raw-string " ")
      (stream/stream (.-dispatchFn f))
      (stream/raw-string "]" {:fill "#ccf"})))
  Fn
  (parse [f]
    (stream/ensure-simple-string
      (Compiler/demunge (.getName (class f)))
      {:fill "#fb6"}))
  Pattern
  (parse [re]
    (stream/horizontal
      (stream/raw-string "#" {:fill "ccf"})
      (stream/ensure-simple-string (str \" re \") {:fill "cec"})))
  Var
  (parse [var]
    (stream/horizontal
      (stream/raw-string (pr-str var) {:fill "#ccf"})))
  IRef
  (parse [*ref]
    (stream/horizontal
      (stream/raw-string (str "#" (.toLowerCase (.getSimpleName (class *ref))) "[") {:fill "#ccf"})
      (identity-hash-code *ref)
      (stream/raw-string " ")
      (stream/stream @*ref)
      (stream/raw-string "]" {:fill "#ccf"})))
  Volatile
  (parse [*ref]
    (stream/horizontal
      (stream/raw-string "#volatile[" {:fill "#ccf"})
      (identity-hash-code *ref)
      (stream/raw-string " ")
      (stream/stream @*ref)
      (stream/raw-string "]" {:fill "#ccf"}))))

(defn prepl-output [x]
  (stream/as x
    (if (:exception x)
      (cond-> (stream/multiline (-> x :val m/ex-triage m/ex-str) {:fill "#f66"})
              (:form x)
              (as-> err-output
                    (stream/vertical
                      (stream/multiline (:form x) {:fill "#999"})
                      err-output)))
      (case (:tag x)
        :ret (stream/vertical
               (stream/multiline (:form x) {:fill "#999"})
               (stream/horizontal
                 (stream/raw-string "=> " {:fill "#999"})
                 (stream/stream (:val x))))
        :out (stream/multiline (:val x) {:fill "#bb6"})
        :err (stream/multiline (:val x) {:fill "#f66"})
        :tap (stream/horizontal
               (stream/raw-string "tap> " {:fill "#999"})
               (stream/stream (:val x)))
        (stream/parse x)))))

(defn action-output [action ret]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill "#999"})
        (stream/stream (:id action)))
      (stream/horizontal
        (stream/raw-string "$> " {:fill "#999"})
        (stream/stream ret)))))

(defn action-error-output [action ex]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill "#999"})
        (stream/stream (:id action)))
      (stream/stream ex nil))))
