(ns reveal.canvas
  (:require [cljfx.lifecycle :as fx.lifecycle]
            [cljfx.mutator :as fx.mutator]
            [cljfx.fx.canvas :as fx.canvas]
            [cljfx.api :as fx]
            [reveal.actions :as actions]
            [cljfx.composite :as fx.composite]
            [reveal.font :as font]
            [reveal.cursor :as cursor]
            [reveal.lines :as lines]
            [reveal.event :as event]
            [clojure.string :as str])
  (:import [javafx.scene.canvas Canvas GraphicsContext]
           [javafx.scene.paint Color]
           [javafx.scene.input ScrollEvent MouseEvent MouseButton KeyEvent KeyCode Clipboard ClipboardContent]))

(set! *warn-on-reflection* true)

(defn- make-resizable-canvas []
  (proxy [Canvas] []
    (isResizable [] true)
    (minWidth [_] 0)
    (minHeight [_] 0)
    (maxWidth [_] Double/MAX_VALUE)
    (maxHeight [_] Double/MAX_VALUE)
    (prefWidth [_]
      (let [^Canvas this this]
        (.getWidth this)))
    (prefHeight [_]
      (let [^Canvas this this]
        (.getHeight this)))
    (resize [w h]
      (let [^Canvas this this]
        (proxy-super setWidth w)
        (proxy-super setHeight h)))))

(def ^:private canvas-lifecycle
  (fx.composite/lifecycle
    {:ctor make-resizable-canvas
     :args []
     :props (merge fx.canvas/props
                   (fx.composite/props Canvas
                     :draw [(fx.mutator/setter
                              (fn [^Canvas canvas [f & args]]
                                (apply f (.getGraphicsContext2D canvas) args)))
                            fx.lifecycle/scalar]
                     :on-width-changed [:property-change-listener fx.lifecycle/change-listener]
                     :on-height-changed [:property-change-listener fx.lifecycle/change-listener]))}))

(defn- clamp [n min-n max-n]
  (-> n
      (max min-n)
      (min max-n)))

(def ^:private ^:const scroll-bar-breadth 10.0)

(def ^:private ^:const min-scroll-tab-size 30.0)

(defn- scroll-tab-size [document-size canvas-size]
  (if (< document-size min-scroll-tab-size)
    min-scroll-tab-size
    (let [visible-ratio (min 1.0 (/ canvas-size document-size))]
      (max min-scroll-tab-size (* canvas-size visible-ratio)))))

(defn- scrolled-to-bottom? [layout]
  (let [{:keys [scroll-y
                canvas-height
                document-height]} layout]
    (or (< document-height canvas-height)
        (= scroll-y (- canvas-height document-height)))))

(defn- scroll-per-pixel [document-size canvas-size scroll-tab-size]
  (let [document-range (- document-size canvas-size)
        scroll-range (- canvas-size scroll-tab-size)]
    (if (zero? scroll-range)
      ##Inf
      (/ document-range scroll-range))))

(defn layout [{:keys [font
                      canvas-width
                      canvas-height
                      lines
                      scroll-x
                      scroll-y] :as layout}]
  (let [line-height (font/line-height font)
        line-count (count lines)
        document-height (+ (* line-height line-count) scroll-bar-breadth)
        scroll-y (clamp scroll-y (- canvas-height document-height) 0.0)
        scroll-y-remainder (rem (- scroll-y) line-height)
        dropped-line-count (- (long (/ scroll-y line-height)))
        drawn-line-count (long (min (- line-count dropped-line-count)
                                    (Math/ceil (/ (+ canvas-height scroll-y-remainder) line-height))))
        document-width (transduce
                         (map #(transduce (comp (mapcat :segments) (map :width)) + 0 (lines %)))
                         max
                         0
                         (range dropped-line-count (+ dropped-line-count drawn-line-count)))
        scroll-x (clamp scroll-x (- canvas-width document-width) 0.0)]
    (-> layout
        (assoc :scroll-x scroll-x
               :scroll-y scroll-y
               :document-height document-height
               :drawn-line-count drawn-line-count
               :dropped-line-count dropped-line-count
               :scroll-y-remainder scroll-y-remainder)
        (as-> $ (if (>= canvas-width document-width)
                  (dissoc $ :scroll-tab-x)
                  (assoc $ :scroll-tab-x (let [visible-left (- scroll-x)
                                               scroll-tab-top (- canvas-height scroll-bar-breadth)
                                               scroll-tab-width (scroll-tab-size document-width canvas-width)
                                               scroll-per-pixel (scroll-per-pixel document-width canvas-width scroll-tab-width)
                                               scroll-tab-left (/ visible-left scroll-per-pixel)]
                                           {:x scroll-tab-left
                                            :y scroll-tab-top
                                            :width scroll-tab-width
                                            :height scroll-bar-breadth
                                            :scroll-per-pixel scroll-per-pixel}))))
        (as-> $ (if (>= canvas-height document-height)
                  (dissoc $ :scroll-tab-y)
                  (assoc $ :scroll-tab-y (let [visible-top (- scroll-y)
                                               scroll-tab-left (- canvas-width scroll-bar-breadth)
                                               scroll-tab-height (scroll-tab-size document-height canvas-height)
                                               scroll-per-pixel (scroll-per-pixel document-height canvas-height scroll-tab-height)
                                               scroll-tab-top (/ visible-top scroll-per-pixel)]
                                           {:x scroll-tab-left
                                            :y scroll-tab-top
                                            :width scroll-bar-breadth
                                            :height scroll-tab-height
                                            :scroll-per-pixel scroll-per-pixel})))))))

(defn- draw-scroll-bar [^GraphicsContext ctx active {:keys [x y width height]}]
  (doto ctx
    (.setFill (Color/valueOf (if active "#fff6" "#eee3")))
    (.fillRoundRect x y width height scroll-bar-breadth scroll-bar-breadth)))

(defn- draw [^GraphicsContext ctx layout]
  (let [{:keys [canvas-height
                canvas-width
                scroll-y-remainder
                ^long drawn-line-count
                ^long dropped-line-count
                scroll-x
                font
                lines
                scroll-tab-x
                scroll-tab-y
                cursor
                anchor
                gesture]} layout
        line-height (font/line-height font)]
    (.clearRect ctx 0 0 canvas-width canvas-height)
    (.setFont ctx (font/jfx-font font))
    (when (and cursor anchor)
      (let [from (-> anchor
                     (cursor/min cursor)
                     (cursor/max [dropped-line-count 0]))
            last-visible-line-index (dec (+ dropped-line-count drawn-line-count))
            to (-> anchor
                   (cursor/max cursor)
                   (cursor/min [last-visible-line-index (dec (count (lines last-visible-line-index)))]))]
        (when-not (cursor/before? to from)
          (.setFill ctx (Color/valueOf "#66a"))
          (doseq [i (range (cursor/row from) (inc (cursor/row to)))]
            (let [line (lines i)
                  start-col (if (= i (cursor/row from))
                              (cursor/col from)
                              0)
                  end-col (if (= i (cursor/row to))
                            (cursor/col to)
                            (dec (count line)))
                  x (transduce
                      (comp
                        (take start-col)
                        (mapcat :segments)
                        (map :width))
                      +
                      scroll-x
                      line)
                  width (transduce
                          (comp
                            (drop start-col)
                            (take (inc (- end-col start-col)))
                            (mapcat :segments)
                            (map :width))
                          +
                          0
                          line)
                  y (- (* line-height (- i dropped-line-count))
                       scroll-y-remainder)]
              (.fillRect ctx x y width line-height))))))
    (dotimes [i drawn-line-count]
      (transduce (mapcat :segments)
                 (completing
                   (fn [x {:keys [text width style]}]
                     (if (< x canvas-width)
                       (let [end (+ x width)]
                         (if (<= end 0)
                           end
                           (do
                             (.setFill ctx (Color/valueOf ^String (:fill style "#000")))
                             (.fillText ctx text x (-> (* i line-height)
                                                       (+ (font/ascent font))
                                                       (- scroll-y-remainder)))
                             end)))
                       (reduced nil))))
                 scroll-x
                 (lines (+ i dropped-line-count))))
    (some->> scroll-tab-x (draw-scroll-bar ctx (= :scroll-x (:type gesture))))
    (some->> scroll-tab-y (draw-scroll-bar ctx (= :scroll-y (:type gesture))))))

(defmacro defevent [multi-sym dispatch-val-expr bindings-expr & body]
  `(let [fn# (fn ~bindings-expr (layout ~@body))]
     (defmethod ~multi-sym ~dispatch-val-expr [event#]
       {:state (update (:state event#) :canvas fn# event#)})))

(defevent event/handle ::on-scroll [layout {:keys [^ScrollEvent fx/event]}]
  (-> layout
      (update :scroll-x + (.getDeltaX event))
      (update :scroll-y + (.getDeltaY event))))

(defevent event/handle ::on-size-changed [layout {:keys [key fx/event]}]
  (assoc layout key event))

(defevent event/handle ::add-lines [layout {:keys [fx/event]}]
  (-> layout
      (update :lines into event)
      (cond-> (scrolled-to-bottom? layout)
              (assoc :scroll-y ##-Inf))))

(defevent event/handle ::on-mouse-released [layout _]
  (dissoc layout :gesture))

(defevent event/handle ::on-window-focus-changed [layout {:keys [fx/event]}]
  (if event layout (dissoc layout :gesture)))

(defn- perform-scroll [layout ^MouseEvent event]
  (if-let [gesture (:gesture layout)]
    (case (:type gesture)
      :scroll-x
      (assoc layout :scroll-x (- (* (- (.getX event) (:offset gesture))
                                    (-> layout :scroll-tab-x :scroll-per-pixel))))
      :scroll-y
      (assoc layout :scroll-y (- (* (- (.getY event) (:offset gesture))
                                    (-> layout :scroll-tab-y :scroll-per-pixel))))

      layout)
    layout))

(defevent event/handle ::on-mouse-dragged [layout {:keys [fx/event]}]
  (perform-scroll layout event))

(defn- empty-region? [region]
  (every? #(-> % :text str/blank?) (:segments region)))

(defn- region-width [region]
  (transduce (map :width) + (:segments region)))

(defn- canvas->cursor [layout x y]
  (let [{:keys [scroll-x scroll-y font lines]} layout
        doc-x (- x scroll-x)
        doc-y (- y scroll-y)
        line-height (font/line-height font)
        row (long (/ doc-y line-height))]
    (when (< row (count lines))
      (let [line (lines row)
            index (first (transduce
                           (map #(region-width (line %)))
                           (completing
                             (fn [[i x] width]
                               (let [x (+ x width)]
                                 (if (< doc-x x)
                                   (reduced [i])
                                   [(inc i) x]))))
                           [0 0]
                           (range (count line))))]
        (when-not (empty-region? (line index))
          [row index])))))

(defevent event/handle ::on-mouse-pressed [layout {:keys [^MouseEvent fx/event]}]
  (cond
    (not= (.getButton event) MouseButton/PRIMARY)
    layout

    (some-> (:scroll-tab-y layout) :x (<= (.getX event)))
    (-> layout
        (assoc :gesture {:type :scroll-y
                         :offset (let [event-y (.getY event)
                                       {:keys [y height]} (:scroll-tab-y layout)]
                                   (if (<= y event-y (+ y height))
                                     (- event-y y)
                                     (* height 0.5)))})
        (perform-scroll event))

    (some-> (:scroll-tab-x layout) :y (<= (.getY event)))
    (-> layout
        (assoc :gesture {:type :scroll-x
                         :offset (let [event-x (.getX event)
                                       {:keys [x width]} (:scroll-tab-x layout)]
                                   (if (<= x event-x (+ x width))
                                     (- event-x x)
                                     (* width 0.5)))})
        (perform-scroll event))

    :else
    (if-let [cursor (canvas->cursor layout (.getX event) (.getY event))]
      (assoc layout :cursor cursor :anchor cursor :align-char-index (:index (get-in (:lines layout) cursor)))
      layout)))

(defn- arrow-scroll [layout size-key]
  (let [{:keys [font]} layout
        line-height (font/line-height font)
        size (get layout size-key)]
    (* line-height
       (-> 5
           (min (Math/ceil (* 0.1 (/ size line-height))))
           (max 1)))))

(defn- page-scroll [layout]
  (let [{:keys [font canvas-height]} layout
        line-height (font/line-height font)]
    (* line-height
       (max 1 (Math/ceil (* 0.5 (/ canvas-height line-height)))))))

(def non-empty-region? (complement empty-region?))

(defn- empty-line? [line]
  (every? empty-region? line))

(def non-empty-line? (complement empty-line?))

(defn- adjust-scroll [scroll canvas-size region-start region-size]
  (let [canvas-start (- scroll)
        region-end (+ region-start region-size)
        canvas-end (+ canvas-start canvas-size)
        start (if (> region-end canvas-end)
                (- region-start (- region-end canvas-end))
                region-start)
        start (if (< start canvas-start)
                (+ start (- canvas-start start))
                start)]
    (- scroll (- region-start start))))

(defn- ensure-cursor-visible [layout]
  (let [{:keys [lines cursor canvas-width canvas-height font]} layout
        [row col] cursor
        line (lines row)
        line-height (font/line-height font)
        region-size (region-width (line col))
        region-start (transduce (map region-width) + (subvec line 0 col))]
    (-> layout
        (update :scroll-y adjust-scroll canvas-height (* line-height (cursor/row cursor)) line-height)
        (update :scroll-x adjust-scroll canvas-width region-start region-size))))

(defn- introduce-cursor-at-bottom-of-screen [layout]
  (let [{:keys [drawn-line-count dropped-line-count lines canvas-height scroll-y-remainder font]} layout
        start-row (cond-> (dec (+ dropped-line-count drawn-line-count))
                          (< canvas-height (- (* drawn-line-count (font/line-height font))
                                              scroll-y-remainder))
                          dec)]
    (if-let [cursor (lines/scan lines [start-row -1] dec inc non-empty-region?)]
      (-> layout
          (assoc :cursor cursor
                 :anchor cursor
                 :align-char-index (:index (get-in lines cursor)))
          ensure-cursor-visible)
      layout)))

(defn- introduce-cursor-at-top-of-screen [layout]
  (let [{:keys [dropped-line-count lines scroll-y-remainder]} layout
        start-row (cond-> dropped-line-count
                          (not (zero? scroll-y-remainder))
                          inc)]
    (if-let [cursor (lines/scan lines [start-row -1] inc inc non-empty-region?)]
      (-> layout
          (assoc :cursor cursor
                 :anchor cursor
                 :align-char-index (:index (get-in lines cursor)))
          ensure-cursor-visible)
      layout)))

(defn- move-cursor-horizontally [layout with-anchor direction]
  (let [{:keys [cursor lines]} layout]
    (if-let [cursor (lines/scan lines cursor direction direction non-empty-region?)]
      (-> layout
          (assoc :cursor cursor :align-char-index (:index (get-in lines cursor)))
          (cond-> with-anchor (assoc :anchor cursor))
          ensure-cursor-visible)
      layout)))

(defn- select-all [layout]
  (let [{:keys [lines]} layout
        from (lines/scan lines [##-Inf ##-Inf] inc inc non-empty-region?)
        to (lines/scan lines [##Inf ##Inf] dec dec non-empty-region?)]
    (cond-> layout
            (and from to)
            (assoc :anchor from :cursor to :align-char-index (:index (get-in lines to))))))

(defn- binary-nearest-by [f xs x]
  (let [last-i (dec (count xs))]
    (loop [low 0
           high last-i]
      (when (<= low high)
        (let [i (quot (+ low high) 2)
              n (f (xs i))]
          (cond
            (and (<= n x)
                 (or (= i last-i)
                     (< x (f (xs (inc i))))))
            i

            (< x n)
            (recur low (dec i))

            :else
            (recur (inc i) high)))))))

(defn- move-cursor-vertically [layout with-anchor direction]
  (let [{:keys [cursor lines align-char-index]} layout
        row (cursor/row cursor)]
    (if-let [row (lines/scan lines row direction non-empty-line?)]
      (let [line (lines row)
            nearest-col (binary-nearest-by :index line align-char-index)
            col (or (some #(when (non-empty-region? (line %)) %)
                          (range nearest-col (count line)))
                    (some #(when (non-empty-region? (line %)) %)
                          (range (dec nearest-col) 0 -1)))
            cursor [row col]]
        (-> layout
            (assoc :cursor cursor)
            (cond-> with-anchor (assoc :anchor cursor))
            ensure-cursor-visible))
      layout)))

(defn- cursor-to-start-of-selection [layout]
  (let [start (cursor/min (:cursor layout) (:anchor layout))]
    (-> layout
        (assoc :cursor start :anchor start)
        ensure-cursor-visible)))

(defn- cursor-to-end-of-selection [layout]
  (let [start (cursor/max (:cursor layout) (:anchor layout))]
    (-> layout
        (assoc :cursor start :anchor start)
        ensure-cursor-visible)))

(defn- cursor-to-end-of-line [layout with-anchor]
  (let [{:keys [lines cursor]} layout
        [row col] cursor
        line (lines row)]
    (if-let [new-col (some #(when (non-empty-region? (line %)) %)
                           (range (dec (count line)) (dec col) -1))]
      (let [cursor [row new-col]]
        (-> layout
            (assoc :cursor cursor :align-char-index (:index (line new-col)))
            (cond-> with-anchor (assoc :anchor cursor))
            ensure-cursor-visible))
      layout)))

(defn- cursor-to-beginning-of-line [layout with-anchor]
  (let [{:keys [lines cursor]} layout
        [row col] cursor
        line (lines row)]
    (if-let [new-col (some #(when (non-empty-region? (line %)) %) (range 0 (inc col)))]
      (let [cursor [row new-col]]
        (-> layout
            (assoc :cursor cursor :align-char-index (:index (line new-col)))
            (cond-> with-anchor (assoc :anchor cursor))
            ensure-cursor-visible))
      layout)))

(defn- string-builder
  ([] (StringBuilder.))
  ([^StringBuilder ret] (.toString ret))
  ([^StringBuilder acc in] (.append acc in)))

(defn- copy-selection! [layout]
  (fx/on-fx-thread
    (let [{:keys [cursor anchor lines]} layout
          from (cursor/min cursor anchor)
          to (cursor/max cursor anchor)
          text (transduce
                 (comp
                   (interpose ::newline)
                   (mapcat (fn [row]
                             (case row
                               ::newline [{:segments [{:text "\n"}]}]
                               (let [line (lines row)
                                     start-col (if (= row (cursor/row from))
                                                 (cursor/col from)
                                                 0)
                                     end-col (if (= row (cursor/row to))
                                               (cursor/col to)
                                               (dec (count line)))]
                                 (subvec line start-col (inc end-col))))))
                   (mapcat :segments)
                   (map :text))
                 string-builder
                 (range (cursor/row from) (inc (cursor/row to))))
          clipboard (Clipboard/getSystemClipboard)
          content (doto (ClipboardContent.)
                    (.putString text))]
      (.setContent clipboard content)))
  layout)

(defn- set-layout-fx [state new-layout]
  {:state (assoc state :canvas (layout new-layout))})

(defmethod event/handle ::on-key-pressed [{:keys [^KeyEvent fx/event state]}]
  (let [layout (:canvas state)
        code (.getCode event)
        shortcut (.isShortcutDown event)
        with-anchor (not (.isShiftDown event))
        {:keys [cursor anchor]} layout]
    (condp = code
      KeyCode/ESCAPE
      (set-layout-fx state (cond-> layout cursor (dissoc :cursor :anchor)))

      KeyCode/UP
      (set-layout-fx state
        (cond
          shortcut (update layout :scroll-y + (arrow-scroll layout :canvas-height))
          (not cursor) (introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (cursor-to-start-of-selection layout)
          :else (move-cursor-vertically layout with-anchor dec)))

      KeyCode/DOWN
      (set-layout-fx state
        (cond
          shortcut (update layout :scroll-y - (arrow-scroll layout :canvas-height))
          (not cursor) (introduce-cursor-at-top-of-screen layout)
          (and with-anchor (not= cursor anchor)) (cursor-to-end-of-selection layout)
          :else (move-cursor-vertically layout with-anchor inc)))

      KeyCode/LEFT
      (set-layout-fx state
        (cond
          shortcut (update layout :scroll-x + (arrow-scroll layout :canvas-width))
          (not cursor) (introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (cursor-to-start-of-selection layout)
          :else (move-cursor-horizontally layout with-anchor dec)))

      KeyCode/RIGHT
      (set-layout-fx state
        (cond
          shortcut (update layout :scroll-x - (arrow-scroll layout :canvas-width))
          (not cursor) (introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (cursor-to-end-of-selection layout)
          :else (move-cursor-horizontally layout with-anchor inc)))

      KeyCode/PAGE_UP
      (set-layout-fx state
        (update layout :scroll-y + (page-scroll layout)))

      KeyCode/PAGE_DOWN
      (set-layout-fx state
        (update layout :scroll-y - (page-scroll layout)))

      KeyCode/HOME
      (set-layout-fx state
        (cond
          shortcut (assoc layout :scroll-y 0)
          (not cursor) (assoc layout :scroll-x 0)
          :else (cursor-to-beginning-of-line layout with-anchor)))

      KeyCode/END
      (set-layout-fx state
        (cond
          shortcut (assoc layout :scroll-y ##-Inf)
          (not cursor) (assoc layout :scroll-x ##-Inf)
          :else (cursor-to-end-of-line layout with-anchor)))

      KeyCode/C
      (set-layout-fx state
        (if (and (.isShortcutDown event) cursor)
          (copy-selection! layout)
          layout))

      KeyCode/A
      (set-layout-fx state
        (if (.isShortcutDown event)
          (select-all layout)
          layout))

      KeyCode/ENTER
      (when cursor
        (let [values (:values (get-in (:lines layout) cursor))]
          (when-let [action (actions/most-fitting values)]
            {:execute action})))

      nil)))

;; TODO
(defmethod event/handle ::on-mouse-entered [_])
(defmethod event/handle ::on-mouse-moved [_])
(defmethod event/handle ::on-mouse-exited [_])

(defn view [{:keys [layout]}]
  {:fx/type canvas-lifecycle
   :draw [draw layout]
   :width (:canvas-width layout)
   :height (:canvas-height layout)
   :focus-traversable true
   :on-key-pressed {::event/type ::on-key-pressed}
   :on-mouse-entered {::event/type ::on-mouse-entered}
   :on-mouse-moved {::event/type ::on-mouse-moved}
   :on-mouse-dragged {::event/type ::on-mouse-dragged}
   :on-mouse-pressed {::event/type ::on-mouse-pressed}
   :on-mouse-released {::event/type ::on-mouse-released}
   :on-mouse-exited {::event/type ::on-mouse-exited}
   :on-width-changed {::event/type ::on-size-changed :key :canvas-width}
   :on-height-changed {::event/type ::on-size-changed :key :canvas-height}
   :on-scroll {::event/type ::on-scroll}})

#_{:tag :ret
   :val {{:a 1
          :x []} {1 0 0 1}
         :b 2}
   :e (RuntimeException. "beep")}

#_(vec (map #(hash-map :n % % % :rand (rand)) (range 10000)))

#_(/ 1 0)
