(ns reveal.stream
  (:refer-clojure :exclude [newline])
  (:require [clojure.string :as str]
            [reveal.font :as font]))

(defprotocol Parse
  :extend-via-metadata true
  (parse* [x]))

(defn- =>
  ([] (fn [_ acc] acc))
  ([f] f)
  ([f g]
   (fn [rf acc]
     (g rf (f rf acc))))
  ([f g & fs]
   (reduce => (cons f (cons g fs)))))

(defn- op [op]
  (fn [rf acc]
    (rf acc op)))

(defn with-value [x annotation sf]
  (=> (op {:op ::push-value :value [x annotation]})
      sf
      (op {:op ::pop-value})))

(defn parse
  ([x]
   (parse x nil))
  ([x annotation]
   (with-value x annotation (parse* x))))

(defn- escaped-string [^String str style]
  (fn [rf acc]
    (let [builder (StringBuilder.)
          len (count str)]
      (loop [i 0
             acc acc]
        (if (= i len)
          (if (zero? (.length builder))
            acc
            (rf acc {:op ::string :text (.toString builder) :style style}))
          (let [ch (.charAt str i)
                e (case ch
                    \newline "\\n"
                    \tab "\\t"
                    \return "\\r"
                    \formfeed "\\f"
                    \backspace "\\b"
                    nil)]
            (recur
              (inc i)
              (if e
                (rf (if (pos? (.length builder))
                      (let [ret (rf acc {:op ::string :text (.toString builder) :style style})]
                        (.delete builder 0 (.length builder))
                        ret)
                      acc)
                    {:op ::string :text e :style {:fill "#f66"}})
                (do (.append builder ch)
                    acc)))))))))

(defn- block [block-type sf]
  (=> (op {:op ::push-block :block block-type})
      sf
      (op {:op ::pop-block})))

(defn- string [str style]
  (op {:op ::string :text str :style style}))

(def ^:private newline
  (op {:op ::newline}))

(defn raw-string
  ([str]
   (raw-string str {}))
  ([str style]
   (block :paragraph (string str style))))

(defn- str+styles-with [str+style->sf str+styles]
  (fn [rf acc]
    (transduce
      (comp
        (partition-all 2)
        (map (fn [[str style]]
               (str+style->sf str style))))
      (completing
        (fn [acc sf]
          (sf rf acc)))
      acc
      str+styles)))

(defn ensure-simple-string
  "Stream str+styles as a single-line, mono-spaced string

  It will escape and highlight as errors all layout-affecting characters, such as \n or \t"
  [& str+styles]
  (block :paragraph (str+styles-with escaped-string str+styles)))

(defn horizontal [& sfs]
  (block :horizontal (apply => sfs)))

(defn vertical [& sfs]
  (block :vertical (apply => sfs)))

(defn entries [m]
  (fn [rf acc]
    (reduce-kv
      (fn [acc k v]
        ((horizontal (parse k {:val v}) (raw-string " ") (parse v {:key k})) rf acc))
      acc
      m)))

(defn- vertical-items [coll]
  (fn [rf acc]
    (let [*i (volatile! -1)]
      (reduce (fn [acc x]
                ((parse x {:index (vswap! *i inc)}) rf acc))
              acc
              coll))))

(defn- horizontal-items [coll]
  (fn [rf acc]
    (let [*i (volatile! -1)]
      (transduce (interpose ::space)
                 (completing
                   (fn [acc x]
                     (if (= x ::space)
                       ((raw-string " ") rf acc)
                       ((parse x {:index (vswap! *i inc)}) rf acc))))
                 acc
                 coll))))

(defn items [coll]
  (if (some coll? coll)
    (block :vertical (vertical-items coll))
    (block :horizontal (horizontal-items coll))))

(defn sequential [xs]
  (block :vertical (vertical-items xs)))

(defn- multiline-string [str style]
  (fn [rf acc]
    (transduce
      (interpose ::newline)
      (completing
        (fn [acc x]
          (if (= x ::newline)
            (newline rf acc)
            ((string x style) rf acc))))
      acc
      (str/split-lines str))))

(defn multiline [str style]
  (block :paragraph (multiline-string str style)))

(defn- parse-xf [rf]
  (fn
    ([] (rf))
    ([result] (rf result))
    ([result input] ((parse* input) rf result))))

(defn- blank-segment [n font]
  {:text (apply str (repeat n \space))
   :width (* n (font/char-width font \space))
   :style {}})

(defn- string-segment [string-op font]
  (-> string-op
      (dissoc :op)
      (assoc :width (transduce (map #(font/char-width font %)) + (:text string-op)))))

(defn- segment-length [segment]
  (count (:text segment)))

(defn- segments-length [segments]
  (transduce (map segment-length) + segments))

(defn- next-index [region]
  (+ (:index region 0)
     (segments-length (:segments region))))

(defn- add-segment [line values segment]
  (let [last-region (peek line)]
    (if (not= (:values last-region) values)
      (conj line {:values values :segments [segment] :index (next-index last-region)})
      (update-in line [(dec (count line)) :segments] conj segment))))

(defn- line-length [line]
  (next-index (peek line)))

(defn- soft-pop [coll]
  (cond-> coll (pos? (count coll)) pop))

(defn- format-xf [font]
  (fn [rf]
    (let [*state (volatile! {:line [] :values [] :blocks []})]
      (fn
        ([] (rf))
        ([result] (rf result))
        ([result input]
         (let [state @*state]
           (case (:op input)
             ::push-value
             (do (vswap! *state update :values conj (:value input))
                 result)

             ::pop-value
             (do (vswap! *state update :values pop)
                 result)

             ::push-block
             (let [blocks (:blocks state)
                   block (peek blocks)]
               (case (:block block)
                 :vertical
                 (if (:had-sub-blocks block)
                   (do (vreset!
                         *state
                         (assoc state
                           :line (add-segment [] (soft-pop (:values state)) (blank-segment (:indent block) font))
                           :blocks (conj blocks {:block (:block input)
                                                 :indent (:indent block)})))
                       (rf result (:line state)))
                   (do (vswap! *state assoc :blocks
                               (-> blocks
                                   (assoc-in [(dec (count blocks)) :had-sub-blocks] true)
                                   (conj {:block (:block input)
                                          :indent (:indent block)})))
                       result))

                 :horizontal
                 (do (vswap! *state update :blocks conj
                             {:block (:block input)
                              :indent (line-length (:line state))})
                     result)

                 nil
                 (do (vswap! *state update :blocks conj
                             {:block (:block input)
                              :indent 0})
                     result)))

             ::pop-block
             (let [blocks (:blocks state)]
               (vswap! *state update :blocks pop)
               (if (= 1 (count blocks))
                 (do (vreset! *state (-> state
                                         (assoc :blocks (pop blocks))
                                         (assoc :line [])))
                     (rf result (:line state)))
                 (do (vreset! *state (assoc state :blocks (pop blocks)))
                     result)))

             ::string
             (do (vswap! *state update :line add-segment (:values state) (string-segment input font))
                 result)

             ::newline
             (let [block (peek (:blocks state))]
               (do (vswap! *state assoc :line (add-segment [] (:values state) (blank-segment (:indent block) font)))
                   (rf result (:line state)))))))))))

(defn stream-xf [font]
  (comp parse-xf (format-xf font)))

#_ (clojure.pprint/pprint (into [] (stream-xf (reveal.font/make "monospace")) [(reveal.parse/prepl-output {:tag :out :val "a"})]))

#_(into [] (stream-xf (reveal.font/make "monospace")) [(ex-info "" {})])