(ns reveal.font
  (:require [cljfx.coerce :as fx.coerce])
  (:import [com.sun.javafx.tk Toolkit]
           [com.sun.javafx.font PGFont FontResource FontStrike]
           [com.sun.javafx.scene.text FontHelper]
           [com.sun.javafx.geom.transform BaseTransform]))

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(deftype Font [font ^double line-height ^double ascent char-width-cache])

(def ^:private ^:const min-cached-char-width
  (inc Double/MIN_VALUE))

(def ^:private ^:const max-cached-char-width
  Double/MAX_VALUE)

(defn- make-char-width-cache [^FontStrike font-strike]
  (let [cache (double-array (inc (int Character/MAX_VALUE)) Double/MIN_VALUE)]
    (fn get-char-width [^Character character]
      (let [ch (unchecked-char character)
            i (unchecked-int ch)
            cached-width (aget cache i)]
        (if (= cached-width Double/MIN_VALUE)
          (let [width (.getCharAdvance font-strike ch)]
            (when (and (<= min-cached-char-width width)
                       (<= width max-cached-char-width))
              (aset cache i width))
            width)
          cached-width)))))

(defn make ^Font [font]
  (let [^javafx.scene.text.Font font (fx.coerce/font font)
        metrics (.getFontMetrics (.getFontLoader (Toolkit/getToolkit)) font)
        strike (.getStrike ^PGFont (FontHelper/getNativeFont font)
                           BaseTransform/IDENTITY_TRANSFORM
                           FontResource/AA_GREYSCALE)]
    (Font. font
           (.getLineHeight metrics)
           (.getAscent metrics)
           (make-char-width-cache strike))))

(defn line-height ^double [font]
  (.-line_height ^Font font))

(defn ascent ^double [font]
  (.-ascent ^Font font))

(defn char-width ^double [font char]
  ((.-char_width_cache ^Font font) char))

(defn jfx-font ^javafx.scene.text.Font [font]
  (.-font ^Font font))
