(ns reveal.view
  (:require [cljfx.lifecycle :as fx.lifecycle]
            [reveal.output-panel :as output-panel]
            [reveal.event :as event]
            [reveal.stream :as stream]
            [reveal.layout :as layout]))

(defprotocol Viewable
  (get-value [this] "Returns value of that viewable")
  (make-view [this env] "Returns cljfx description for the viewable"))

(defn- runduce!
  ([xf x]
   (runduce! xf identity x))
  ([xf f x]
   (let [rf (xf (completing #(f %2)))]
     (rf (rf nil x)))))

(defn value [value env]
  ((:on-create env)
   (fn [dispatch!]
     (let [t (Thread.
               ^Runnable
               (fn []
                 (try
                   (runduce! (comp stream/stream-xf (partition-all 128))
                             #(if (.isInterrupted (Thread/currentThread))
                                (reduced nil)
                                (dispatch! {::event/handler output-panel/on-add-lines
                                            :fx/event %
                                            :id (:id env)}))
                             value)
                   (catch InterruptedException _)))
               "reveal-output-panel-value-thread")]
       (.start t)
       #(.interrupt t))))
  {:fx/type output-panel/view
   :id (:id env)
   :layout (layout/make)})

(extend-protocol Viewable
  nil
  (get-value [this] this)
  (make-view [this env] (value this env))

  Object
  (get-value [this] this)
  (make-view [this env] (value this env)))

;; resolution:
;; - views (separate from cljfx component graph!)
;; - view is a cljfx description, with implicit conversion for every value to default
;;   view, and javafx nodes to cljfx descriptions, and treating maps with :fx/type keys as
;;   cljfx descriptions
;; - compose cljfx and javafx by making a wrapper that transforms cljfx components to
;;   javafx nodes, with easy way to pass all necessary information for submitting results
;;   back to main app (can be a function that should be called to submit results)
;; - result is a bad name: not related to action, it's a value with a view (viewable? showable?)
;;   why is it NOT related to action? because it can be submitted to repl output instead of accordion?
;; - value+view "results" from actions can be forced into repl output panel (alt+enter
;;   instead of enter) with custom action ("view")

;; start/stop should work in javafx nodes as well, we have to make sure stop is called at
;; some point when created javafx node is no longer used.
;; possible options:
;; - pass "env" from entry point to created nodes, nodes start when created, then register
;;   themselves in this env and disable themselves when stopped (notified by env). upsides...

;; components should keep running even when not shown, instead of start/stop semantics we
;; actually need just dispose semantics.

;; undecided:
;; - ToCljfx protocol
;; - start/stop semantics on what level: javafx (added-removed), cljfx (created/deleted)
;;   or some other (when? should also work in independent cljfx components...)

;; why cljfx desc instead of javafx by default:
;; - don't require cljfx/javafx dependency if can describe view with data like {:fx/type :label :text "hi"}

;; then node api is of input and event handler...

;; can call events on state even before component is put into output panel...

;; example views (in addition to "as value"):
;; - atom: as watcher (always show latest), as log (show every successor value)
;; - url: as get content (may be json view or web page)
;; - directory: as file tree
;; - file: as content
;; - javafx node: as node
;; - spec: as spec form
;; - stack trace: as table
;; - coll of maps with similar keys: as table
;; - coll of vectors of same length: as table
;; - documentable things: as documentation markdown
;; - sourceable things: as source code

;; #reveal/atom[...] → Watch
;; {:k :v}

;; 0x162543 → Show
;; 6842322

;; 64217538987665 → Labels
;; (:_id :_declared-properties :build-deps)

;; :build-deps → Value
;; {:x 1 :build-fn inc}

;; inc → Inspect
;; {class clojure.core.inc}
