(ns reveal.core
  (:require [clojure.core.server :as server]
            [clojure.edn :as edn]
            [clojure.main :as m]
            [reveal.stream :as stream]
            [reveal.writer-output-stream :as writer-output-stream]
            [reveal.ui :as ui]
            [reveal.style :as style]
            [clojure.string :as str])
  (:import [java.io PrintStream]))

(defn- line-print-stream [line-fn]
  (let [sb (StringBuilder.)]
    (-> #(doseq [^char ch %]
           (if (= \newline ch)
             (let [str (.toString sb)]
               (.delete sb 0 (.length sb))
               (line-fn str))
             (.append sb ch)))
        (PrintWriter-on nil)
        (writer-output-stream/make)
        (PrintStream. true "UTF-8"))))

(defn- wrap-err-out [f ui]
  (fn []
    (let [out System/out
          err System/err]
      (System/setOut (line-print-stream #(do
                                           (.println out %)
                                           (-> % stream/system-out ui))))
      (System/setErr (line-print-stream #(do (.println err %)
                                             (-> % stream/system-err ui))))
      (f)
      (System/setOut out)
      (System/setErr err))))

(defn prepl-output [x]
  (stream/as x
    (if (:exception x)
      (cond-> (stream/raw-string (-> x :val m/ex-triage m/ex-str) {:fill ::style/error-color})
              (:form x)
              (as-> err-output
                    (stream/vertical
                      (stream/raw-string (:form x) {:fill ::style/util-color})
                      err-output)))
      (case (:tag x)
        :ret (stream/vertical
               (stream/raw-string (:form x) {:fill ::style/util-color})
               (stream/horizontal
                 (stream/raw-string "=>" {:fill ::style/util-color})
                 stream/separator
                 (stream/raw-string " " {:fill ::style/util-color})
                 (stream/stream (:val x))))
        :out (stream/raw-string (str/trim-newline (:val x)) {:fill ::style/string-color})
        :err (stream/raw-string (str/trim-newline (:val x)) {:fill ::style/error-color})
        :tap (stream/horizontal
               (stream/raw-string "tap>" {:fill ::style/util-color})
               stream/separator
               (stream/raw-string " " {:fill ::style/util-color})
               (stream/stream (:val x)))
        (stream/emit x)))))

(defn start [& {:keys [prepl streams]
                :or {prepl server/prepl
                     streams true}}]
  (let [out *out*
        err *err*
        ui (ui/make)
        repl (-> #(prepl *in* (fn [x]
                                (ui (prepl-output x))
                                (binding [*out* out]
                                  (if (:exception x)
                                    (binding [*out* err]
                                      (println (m/ex-str (m/ex-triage (:val x))))
                                      (flush))
                                    (do
                                      (case (:tag x)
                                        :out (println (:val x))
                                        :err (binding [*out* err]
                                               (println (:val x)))
                                        (:tap :ret) (prn (:val x))
                                        (prn x))
                                      (flush)))
                                  (print (str (:ns x *ns*) "=> "))
                                  (flush))))
                 (cond-> streams (wrap-err-out ui)))
        v (str "Clojure " (clojure-version))]
    (ui (stream/as *clojure-version*
          (stream/raw-string v {:fill ::style/util-color})))
    (println v)
    (print (str (.name *ns*) "=> "))
    (flush)
    (repl)
    nil))

#_(:document-height (:layout (val (first (filter (comp uuid? key) @-state)))))

#_(do
    (add-watch -state :focus (fn [_ _ o n]
                               (let [old (for [[k v] o
                                               :when (:layout v)]
                                           [k (:focused (:layout v))])
                                     new (for [[k v] n
                                               :when (:layout v)]
                                           [k (:focused (:layout v))])]
                                 (when-not (= old new)
                                   (tap> {:old old
                                          :new new
                                          :trace (:trace (Throwable->map (RuntimeException.)))})))))
    nil)

#_(for [[k v] @-state
        :when (:layout v)]
    [k (keys (:layout v))])

#_(require 'criterium.core)

#_(let [v [{:a 1 :b 2 :c (RuntimeException.)}]
        xf stream/stream-xf
        rf (constantly nil)]
    (criterium.core/quick-bench
      (transduce xf rf nil v))
    (flush))

; protocol-based version
; Evaluation count : 504 in 6 samples of 84 calls.
;             Execution time mean : 1.246660 ms
;    Execution time std-deviation : 18.425994 µs
;   Execution time lower quantile : 1.225961 ms ( 2.5%)
;   Execution time upper quantile : 1.265027 ms (97.5%)
;                   Overhead used : 8.216258 ns

#_(-> -state deref :output :layout keys)

#_(start)

#_(/ 1 0)

#_(future
    (Thread/sleep 1000)
    (prn (:focused (:layout (:output @-state)))))

#_(.start (Thread. #(prn 1)))

#_(do
    (require 'criterium.core)
    (criterium.core/quick-bench (+ 1 2)))

#_(range 10000)

#_(do :repl/quit)

#_(.printStackTrace (RuntimeException.))

(defn -main [& args]
  (apply start (map edn/read-string args)))

#_(stream/as 1 (stream/raw-string "aaa\tb" {:fill "#ccc"}))

#_(font/char-width \a)
#_font/line-height
