(ns reveal.popup
  (:require [reveal.event :as event]
            [cljfx.fx.popup :as fx.popup]
            [reveal.search :as search]
            [cljfx.composite :as fx.composite]
            [cljfx.lifecycle :as fx.lifecycle]
            [cljfx.mutator :as fx.mutator]
            [reveal.style :as style]
            [cljfx.api :as fx]
            [cljfx.ext.list-view :as fx.ext.list-view]
            [reveal.view :as view]
            [reveal.layout :as layout]
            [reveal.stream :as stream])
  (:import [javafx.geometry Bounds Rectangle2D]
           [javafx.stage Screen Popup]
           [com.sun.javafx.event RedirectedEvent]
           [javafx.event Event]
           [javafx.scene.input KeyCode KeyEvent]
           [java.util Collection]
           [javafx.scene.control ListView]))

;; TODO
;; - results are displayed in same popup (enter)
;; - results can be added to main output with closing popup (alt+enter)
;; - results can be cljfx markup

(defn- consume-popup-event [^Event e]
  (if (instance? RedirectedEvent e)
    (.consume (.getOriginalEvent ^RedirectedEvent e))
    (.consume e)))

(defmethod event/handle ::on-key-pressed [{:keys [^KeyEvent fx/event on-cancel font state path]}]
  (condp = (.getCode event)
    KeyCode/ESCAPE
    (if (empty? (:filter-text (get-in state path)))
      {:dispatch on-cancel}
      {:state (update-in state path dissoc :filter-text)})

    KeyCode/ENTER
    (when-let [action (.getFocusedItem (.getFocusModel ^ListView (.getTarget event)))]
      ;; todo handle exceptions better
      ;; todo stream output
      ;; todo eval
      {:state (update-in state path assoc :output
                         {:layout (-> {:font font
                                       :canvas-width 0.0
                                       :canvas-height 0.0
                                       :lines []}
                                      (layout/make)
                                      (layout/add-lines (into []
                                                              (stream/stream-xf font)
                                                              [((:invoke action))])))})})
    nil))

(defmethod event/handle ::on-result-key-pressed [{:keys [^KeyEvent fx/event on-cancel]}]
  (when (= KeyCode/ESCAPE (.getCode event))
    {:dispatch on-cancel}))

(defmethod event/handle ::on-key-typed [{:keys [^KeyEvent fx/event state path]}]
  (let [ch (.getCharacter event)
        path (conj path :filter-text)
        ^String text (get-in state path "")]
    (cond
      (.isEmpty ch)
      nil

      (= 27 (int (.charAt ch 0)))
      nil

      (= "\r" ch)
      nil

      (and (= "\b" ch) (not (.isEmpty text)))
      {:state (assoc-in state path (subs text 0 (dec (.length text))))}

      (= "\b" ch)
      nil

      :else
      {:state (assoc-in state path (str text ch))})))

(defmethod event/handle ::on-selected-action-changed [{:keys [fx/event state path]}]
  {:state (update-in state path assoc :selected-action event)})

(def ^:private lifecycle
  (fx.composite/describe Popup
    :ctor []
    :props (merge fx.popup/props
                  (fx.composite/props Popup
                    :stylesheets [(fx.mutator/setter
                                    (fn [^Popup popup ^Collection styles]
                                      (.setAll (.getStylesheets (.getScene popup)) styles)))
                                  fx.lifecycle/scalar
                                  :default []]))))

;; todo popup-view API:
;; - on cancel (closes popup)
;; - on commit (closes popup, "returns value"???)
;; - on select (replaces popup view)

;; parts of popup: bounds, path

(defn- view-impl [{:keys [actions
                          ^Bounds bounds
                          path
                          on-cancel
                          reveal/css
                          filter-text
                          selected-action
                          output
                          font]
                   :or {filter-text ""
                        output ::no-output}}]
  (let [actions (cond-> actions
                        (not= "" filter-text)
                        (search/select filter-text :label))
        ^Screen screen (first (Screen/getScreensForRectangle (.getMinX bounds)
                                                             (.getMinY bounds)
                                                             (.getWidth bounds)
                                                             (.getHeight bounds)))
        screen-bounds (.getVisualBounds screen)
        bounds-min-x (max (.getMinX screen-bounds) (.getMinX bounds))
        bounds-min-y (max (.getMinY screen-bounds) (.getMinY bounds))
        bounds (Rectangle2D. bounds-min-x
                             bounds-min-y
                             (- (min (.getMaxX screen-bounds) (.getMaxX bounds))
                                bounds-min-x)
                             (- (min (.getMaxY screen-bounds) (.getMaxY bounds))
                                bounds-min-y))
        content-width 300 ;; todo dynamic? resizable?
        shadow-radius 10
        shadow-offset-y 5
        popup-width (+ content-width shadow-radius shadow-radius)
        space-below (- (.getMaxY screen-bounds)
                       (.getMaxY bounds))
        space-above (- (.getMinY bounds)
                       (.getMinY screen-bounds))
        popup-at-the-bottom (< space-above space-below)
        pref-anchor-x (-> (.getMinX bounds)
                          (+ (* (.getWidth bounds) 0.5))
                          (- (* popup-width 0.5)))
        visible-start-x (+ pref-anchor-x shadow-radius)
        visible-end-x (+ pref-anchor-x popup-width (- shadow-radius))
        anchor-fix-x (cond
                       (< visible-start-x (.getMinX screen-bounds))
                       (- (.getMinX screen-bounds) visible-start-x)

                       (> visible-end-x (.getMaxX screen-bounds))
                       (- (.getMaxX screen-bounds) visible-end-x)

                       :else
                       0)
        arrow-width 10
        arrow-height 10
        arrow-x (- (* content-width 0.5) anchor-fix-x)
        max-content-height (- (if popup-at-the-bottom
                                space-below
                                space-above)
                              arrow-height)]
    {:fx/type lifecycle
     :stylesheets [(:cljfx.css/url style/dark)]
     :anchor-location (if popup-at-the-bottom :window-top-left :window-bottom-left)
     :anchor-x (+ pref-anchor-x anchor-fix-x)
     :anchor-y (if popup-at-the-bottom
                 (- (.getMaxY bounds) shadow-radius (- shadow-offset-y))
                 (+ (.getMinY bounds) shadow-radius shadow-offset-y))
     :auto-fix false
     :hide-on-escape false
     :event-handler consume-popup-event
     :content
     [{:fx/type :v-box
       :pref-width content-width
       :max-width content-width
       :effect {:fx/type :drop-shadow
                :radius shadow-radius
                :offset-y shadow-offset-y
                :color "#0006"}
       :children
       (-> []
           (cond-> popup-at-the-bottom
                   (conj {:fx/type :polygon
                          :v-box/margin {:left (- arrow-x (* arrow-width 0.5))}
                          :fill (::style/popup-color css)
                          :points [0 arrow-height
                                   arrow-width arrow-height
                                   (* arrow-width 0.5) 0]}))
           (conj
             (if (= output ::no-output)
               {:fx/type :stack-pane
                :style-class "reveal-popup"
                :children
                (cond-> [{:fx/type fx.ext.list-view/with-selection-props
                          :props {:selected-item (or (and selected-action
                                                          (some #(when (= % selected-action) %) actions))
                                                     (first actions))
                                  :on-selected-item-changed {::event/type ::on-selected-action-changed :path path}}
                          :desc {:fx/type :list-view
                                 :style-class "reveal-popup-list-view"
                                 :placeholder {:fx/type :label
                                               :style-class "reveal-placeholder"
                                               :text "No actions match"}
                                 :pref-height (+ (::style/scroll-bar-size css)
                                                 1
                                                 (* (count actions)
                                                    (::style/cell-height css)))
                                 :on-key-typed {::event/type ::on-key-typed :path path}
                                 :on-key-pressed {::event/type ::on-key-pressed :path path :on-cancel on-cancel :font font}
                                 :max-height max-content-height
                                 :cell-factory (fn [action]
                                                 {:text (:label action)})
                                 :items actions}}]
                        (pos? (count filter-text))
                        (conj {:fx/type :label
                               :mouse-transparent true
                               :stack-pane/alignment (if popup-at-the-bottom :top-right :bottom-right)
                               :style-class "search-label"
                               :text filter-text}))}
               {:fx/type :v-box
                ;; TODO padding as a part of layout
                :max-height max-content-height
                :style-class "reveal-popup"
                :on-key-pressed {::event/type ::on-result-key-pressed :on-cancel on-cancel}
                :children [(assoc output :fx/type view/view
                                         :v-box/vgrow :always
                                         ::view/view :reveal.output-panel/view
                                         :path (conj path :output))]}))

           (cond-> (not popup-at-the-bottom)
                   (conj {:fx/type :polygon
                          :v-box/margin {:left (- arrow-x (* arrow-width 0.5))}
                          :fill (::style/popup-color css)
                          :points [0 0
                                   arrow-width 0
                                   (* arrow-width 0.5) arrow-height]})))}]}))

(defmethod view/view ::view [props]
  {:fx/type fx/ext-get-env
   :env [:reveal/css]
   :desc (assoc props :fx/type view-impl)})
