(ns reveal.parse
  (:require [reveal.stream :as stream]
            [reveal.style :as style]
            [clojure.main :as m])
  (:import [clojure.lang Keyword Symbol IPersistentMap IPersistentVector IPersistentSet Fn IPersistentList ISeq MultiFn
                         IRef Var Volatile Namespace IRecord Delay IBlockingDeref TaggedLiteral Reduced ReaderConditional]
           [java.util.regex Pattern]
           [java.io File]
           [java.net URL URI]))

(set! *warn-on-reflection* true)

(defn- identity-hash-code [x]
  (let [hash (System/identityHashCode x)]
    (stream/as hash
      (stream/raw-string (format "0x%x" hash) {:fill ::style/scalar-color}))))

(defmethod stream/parse nil [x]
  (stream/raw-string (pr-str x) {:fill ::style/scalar-color}))

(defmethod stream/parse Boolean [x]
  (stream/raw-string (pr-str x) {:fill ::style/scalar-color}))

(defmethod stream/parse String [s]
  (stream/raw-string (pr-str s) {:fill ::style/string-color}))

(defmethod stream/parse Character [ch]
  (stream/raw-string (pr-str ch) {:fill ::style/string-color}))

(def escape-layout-chars
  {\newline "\\n"
   \tab "\\t"
   \return "\\r"
   \formfeed "\\f"
   \backspace "\\b"})

(defmethod stream/parse Keyword [k]
  (stream/escaped-string k {:fill ::style/keyword-color}
                         escape-layout-chars {:fill ::style/scalar-color}))

(defmethod stream/parse Symbol [sym]
  (stream/escaped-string sym {:fill ::style/symbol-color}
                         escape-layout-chars {:fill ::style/scalar-color}))

(defmethod stream/parse Number [n]
  (stream/raw-string n {:fill ::style/scalar-color}))

(defmethod stream/parse Float [n]
  (stream/raw-string
    (cond
      (= Float/POSITIVE_INFINITY n) "##Inf"
      (= Float/NEGATIVE_INFINITY n) "##-Inf"
      (Float/isNaN n) "##NaN"
      :else (str n))
    {:fill ::style/scalar-color}))

(defmethod stream/parse Double [n]
  (stream/raw-string
    (cond
      (= Double/POSITIVE_INFINITY n) "##Inf"
      (= Double/NEGATIVE_INFINITY n) "##-Inf"
      (Double/isNaN n) "##NaN"
      :else (str n))
    {:fill ::style/scalar-color}))

(defmethod stream/parse IPersistentMap [m]
  (stream/horizontal
    (stream/raw-string "{" {:fill ::style/object-color})
    (stream/vertical (stream/entries m))
    (stream/raw-string "}" {:fill ::style/object-color})))

(defmethod stream/parse IRecord [m]
  (stream/horizontal
    (stream/raw-string (str "#" (.getName (class m)) "{") {:fill ::style/object-color})
    (stream/vertical (stream/entries m))
    (stream/raw-string "}" {:fill ::style/object-color})))

(prefer-method stream/parse IRecord IPersistentMap)
(prefer-method stream/parse ISeq IPersistentList)

(defmethod stream/parse IPersistentVector [v]
  (stream/horizontal
    (stream/raw-string "[" {:fill ::style/object-color})
    (stream/items v)
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse IPersistentList [v]
  (stream/horizontal
    (stream/raw-string "(" {:fill ::style/object-color})
    (stream/items v)
    (stream/raw-string ")" {:fill ::style/object-color})))

(defmethod stream/parse ISeq [xs]
  (stream/horizontal
    (stream/raw-string "(" {:fill ::style/object-color})
    (stream/sequential xs)
    (stream/raw-string ")" {:fill ::style/object-color})))

(defmethod stream/parse IPersistentSet [s]
  (stream/horizontal
    (stream/raw-string "#{" {:fill ::style/object-color})
    (stream/items s)
    (stream/raw-string "}" {:fill ::style/object-color})))

(defmethod stream/parse Throwable [t]
  (stream/horizontal
    (stream/raw-string "#reveal/error" {:fill ::style/object-color})
    (stream/stream (Throwable->map t))))

(defmethod stream/parse MultiFn [^MultiFn f]
  (stream/horizontal
    (stream/raw-string "#reveal/multi-fn[" {:fill ::style/object-color})
    (identity-hash-code f)
    (stream/raw-string " ")
    (stream/stream (.-dispatchFn f))
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse Fn [f]
  (stream/escaped-string (Compiler/demunge (.getName (class f))) {:fill ::style/object-color}
                         escape-layout-chars {:fill ::style/scalar-color}))

(defmethod stream/parse Pattern [re]
  (stream/horizontal
    (stream/raw-string "#" {:fill ::style/object-color})
    (stream/raw-string (str \" re \") {:fill ::style/string-color})))

(defmethod stream/parse Var [var]
  (stream/raw-string (pr-str var) {:fill ::style/object-color}))

(defmethod stream/parse Namespace [^Namespace ns]
  (stream/raw-string (name (.getName ns)) {:fill ::style/object-color}))

(defmethod stream/parse Class [^Class class]
  (stream/raw-string (.getName class) {:fill ::style/object-color}))

(defmethod stream/parse Enum [^Enum enum]
  (stream/raw-string (str (.getName (.getDeclaringClass enum)) "/" (.name enum))
                     {:fill ::style/scalar-color}))

(defmethod stream/parse IRef [*ref]
  (stream/horizontal
    (stream/raw-string (str "#" (.toLowerCase (.getSimpleName (class *ref))) "[") {:fill ::style/object-color})
    (identity-hash-code *ref)
    (stream/raw-string " ")
    (stream/stream @*ref)
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse File [file]
  (stream/horizontal
    (stream/raw-string "#reveal/file[" {:fill ::style/object-color})
    stream/separator
    (stream/escaped-string file {:fill ::style/string-color}
                           escape-layout-chars {:fill ::style/scalar-color})
    stream/separator
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse Delay [*delay]
  (stream/horizontal
    (stream/raw-string "#reveal/delay[" {:fill ::style/object-color})
    (identity-hash-code *delay)
    (stream/raw-string " ")
    (if (realized? *delay)
      (stream/stream @*delay)
      (stream/raw-string "..." {:fill ::style/util-color}))
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse Reduced [*reduced]
  (stream/horizontal
    (stream/raw-string "#reveal/reduced" {:fill ::style/object-color})
    stream/separator
    (stream/raw-string " ")
    (stream/stream @*reduced)))

(defmethod stream/parse IBlockingDeref [*blocking-deref]
  (let [class-name (.getName (class *blocking-deref))]
    (cond
      (.startsWith class-name "clojure.core$promise$reify")
      (stream/horizontal
        (stream/raw-string "#reveal/promise[" {:fill ::style/object-color})
        (identity-hash-code *blocking-deref)
        (stream/raw-string " ")
        (if (realized? *blocking-deref)
          (stream/stream @*blocking-deref)
          (stream/raw-string "..." {:fill ::style/util-color}))
        (stream/raw-string "]" {:fill ::style/object-color}))

      (.startsWith class-name "clojure.core$future_call$reify")
      (stream/horizontal
        (stream/raw-string "#reveal/future[" {:fill ::style/object-color})
        (identity-hash-code *blocking-deref)
        (stream/raw-string " ")
        (if (realized? *blocking-deref)
          (stream/stream @*blocking-deref)
          (stream/raw-string "..." {:fill ::style/util-color}))
        (stream/raw-string "]" {:fill ::style/object-color}))

      :else
      (stream/raw-string (pr-str *blocking-deref) {:fill ::style/object-color}))))

(defmethod stream/parse Volatile [*ref]
  (stream/horizontal
    (stream/raw-string "#reveal/volatile[" {:fill ::style/object-color})
    (identity-hash-code *ref)
    (stream/raw-string " ")
    (stream/stream @*ref)
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse TaggedLiteral [x]
  (stream/horizontal
    (stream/raw-string "#" {:fill ::style/object-color})
    (stream/stream (:tag x))
    (stream/raw-string " ")
    (stream/stream (:form x))))

(defmethod stream/parse ReaderConditional [^ReaderConditional x]
  (stream/horizontal
    (stream/raw-string (str "#?" (when (.-splicing x) "@")) {:fill ::style/object-color})
    (stream/stream (.-form x))))

(defmethod stream/parse URL [x]
  (stream/horizontal
    (stream/raw-string "#reveal/url[" {:fill ::style/object-color})
    stream/separator
    (stream/escaped-string x {:fill ::style/string-color}
                           escape-layout-chars {:fill ::style/scalar-color})
    stream/separator
    (stream/raw-string "]" {:fill ::style/object-color})))

(defmethod stream/parse URI [x]
  (stream/horizontal
    (stream/raw-string "#reveal/uri[" {:fill ::style/object-color})
    stream/separator
    (stream/escaped-string x {:fill ::style/string-color}
                           escape-layout-chars {:fill ::style/scalar-color})
    stream/separator
    (stream/raw-string "]" {:fill ::style/object-color})))

(defn prepl-output [x]
  (stream/as x
    (if (:exception x)
      (cond-> (stream/raw-string (-> x :val m/ex-triage m/ex-str) {:fill ::style/error-color})
              (:form x)
              (as-> err-output
                    (stream/vertical
                      (stream/raw-string (:form x) {:fill ::style/util-color})
                      err-output)))
      (case (:tag x)
        :ret (stream/vertical
               (stream/raw-string (:form x) {:fill ::style/util-color})
               (stream/horizontal
                 (stream/raw-string "=>" {:fill ::style/util-color})
                 stream/separator
                 (stream/raw-string " " {:fill ::style/util-color})
                 (stream/stream (:val x))))
        :out (stream/raw-string (:val x) {:fill ::style/string-color})
        :err (stream/raw-string (:val x) {:fill ::style/error-color})
        :tap (stream/horizontal
               (stream/raw-string "tap>" {:fill ::style/util-color})
               stream/separator
               (stream/raw-string " " {:fill ::style/util-color})
               (stream/stream (:val x)))
        (stream/parse x)))))

(defn system-out [line]
  (stream/as line
    (stream/raw-string line {:fill ::style/string-color})))

(defn system-err [line]
  (stream/as line
    (stream/raw-string line {:fill ::style/error-color})))
