(ns reveal.output-panel
  (:require [reveal.event :as event]
            [reveal.layout :as layout]
            [reveal.actions :as actions]
            [reveal.popup :as popup]
            [reveal.view :as view]
            [cljfx.api :as fx]
            [reveal.canvas :as canvas])
  (:import [javafx.scene.input ScrollEvent KeyEvent MouseEvent MouseButton KeyCode Clipboard ClipboardContent]
           [javafx.scene.canvas Canvas]))

(defn- update-state-fx [state path f & args]
  {:state (apply update-in state path f args)})

(defmethod event/handle ::on-scroll [{:keys [state path ^ScrollEvent fx/event]}]
  (update-state-fx state path update :layout
                   layout/scroll-by (.getDeltaX event) (.getDeltaY event)))

(defmethod event/handle ::on-width-changed [{:keys [state path fx/event]}]
  (update-state-fx state path update :layout layout/set-canvas-width event))

(defmethod event/handle ::on-height-changed [{:keys [state path fx/event]}]
  (update-state-fx state path update :layout layout/set-canvas-height event))

(defmethod event/handle ::add-lines [{:keys [state path fx/event]}]
  (update-state-fx state path update :layout layout/add-lines event))

(defmethod event/handle ::on-mouse-released [{:keys [state path]}]
  (update-state-fx state path update :layout layout/stop-gesture))

(defmethod event/handle ::on-window-focus-changed [{:keys [fx/event state path]}]
  (when-not event
    (update-state-fx state path update :layout layout/stop-gesture)))

(defmethod event/handle ::on-mouse-dragged [{:keys [fx/event state path]}]
  (update-state-fx state path update :layout layout/perform-scroll event))

(defn- show-popup [output-panel ^KeyEvent event]
  (let [layout (layout/ensure-cursor-visible (:layout output-panel))
        {:keys [lines cursor]} layout
        actions (actions/collect (:values (get-in lines cursor)))
        bounds (layout/cursor->canvas-bounds layout)
        ^Canvas target (.getTarget event)
        screen-bounds (.localToScreen target bounds)]
    (-> output-panel
        (assoc :layout layout)
        (cond-> (pos? (count actions))
                (assoc :popup {:actions actions :bounds screen-bounds})))))

(defn- handle-mouse-pressed [output-panel ^MouseEvent event]
  (cond
    (= (.getButton event) MouseButton/SECONDARY)
    (let [layout (:layout output-panel)]
      (if-let [cursor (layout/canvas->cursor layout (.getX event) (.getY event))]
        (-> output-panel
            (assoc :layout (layout/set-cursor layout cursor))
            (show-popup event))
        output-panel))

    :else
    (update output-panel :layout layout/start-gesture event)))

(defmethod event/handle ::on-mouse-pressed [{:keys [fx/event state path]}]
  (update-state-fx state path handle-mouse-pressed event))

(defn- copy-selection! [layout]
  (fx/on-fx-thread
    (.setContent (Clipboard/getSystemClipboard)
                 (doto (ClipboardContent.)
                   (.putString (layout/selection-as-text layout)))))
  layout)

(defn- handle-key-pressed [output-panel ^KeyEvent event]
  (let [code (.getCode event)
        shortcut (.isShortcutDown event)
        with-anchor (not (.isShiftDown event))
        layout (:layout output-panel)
        {:keys [cursor anchor]} layout]
    (condp = code
      KeyCode/ESCAPE
      (assoc output-panel
        :layout
        (cond-> layout cursor layout/remove-cursor))

      KeyCode/UP
      (assoc output-panel
        :layout
        (cond
          shortcut (layout/arrow-scroll-up layout)
          (not cursor) (layout/introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (layout/cursor-to-start-of-selection layout)
          :else (layout/move-cursor-vertically layout with-anchor dec)))

      KeyCode/DOWN
      (assoc output-panel
        :layout
        (cond
          shortcut (layout/arrow-scroll-down layout)
          (not cursor) (layout/introduce-cursor-at-top-of-screen layout)
          (and with-anchor (not= cursor anchor)) (layout/cursor-to-end-of-selection layout)
          :else (layout/move-cursor-vertically layout with-anchor inc)))

      KeyCode/LEFT
      (assoc output-panel
        :layout
        (cond
          shortcut (layout/arrow-scroll-left layout)
          (not cursor) (layout/introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (layout/cursor-to-start-of-selection layout)
          :else (layout/move-cursor-horizontally layout with-anchor dec)))

      KeyCode/RIGHT
      (assoc output-panel
        :layout
        (cond
          shortcut (layout/arrow-scroll-right layout)
          (not cursor) (layout/introduce-cursor-at-bottom-of-screen layout)
          (and with-anchor (not= cursor anchor)) (layout/cursor-to-end-of-selection layout)
          :else (layout/move-cursor-horizontally layout with-anchor inc)))

      KeyCode/PAGE_UP
      (assoc output-panel :layout (layout/page-scroll-up layout))

      KeyCode/PAGE_DOWN
      (assoc output-panel :layout (layout/page-scroll-down layout))

      KeyCode/HOME
      (assoc output-panel
        :layout
        (cond
          shortcut (-> layout
                       layout/scroll-to-top
                       (cond-> cursor layout/remove-cursor))
          (not cursor) (layout/scroll-to-left layout)
          :else (layout/cursor-to-beginning-of-line layout with-anchor)))

      KeyCode/END
      (assoc output-panel
        :layout
        (cond
          shortcut (-> layout
                       layout/scroll-to-bottom
                       (cond-> cursor layout/remove-cursor))
          (not cursor) (layout/scroll-to-right layout)
          :else (layout/cursor-to-end-of-line layout with-anchor)))

      KeyCode/C
      (assoc output-panel
        :layout
        (if (and (.isShortcutDown event) cursor)
          (copy-selection! layout)
          layout))

      KeyCode/A
      (assoc output-panel
        :layout
        (if (.isShortcutDown event)
          (layout/select-all layout)
          layout))

      KeyCode/SPACE
      (cond-> output-panel cursor (show-popup event))

      output-panel)))

(defmethod event/handle ::on-key-pressed [{:keys [^KeyEvent fx/event state path]}]
  (update-state-fx state path handle-key-pressed event))

(defmethod event/handle ::hide-popup [{:keys [state path]}]
  (update-state-fx state path dissoc :popup))

(defn- view-impl [{:keys [layout path popup reveal/css]}]
  (let [{:keys [canvas-width canvas-height document-height]} layout]
    (cond-> {:fx/type canvas/view
             :draw [layout/draw css layout]
             :width canvas-width
             :height canvas-height
             :pref-height document-height
             :focus-traversable true
             :on-key-pressed {::event/type ::on-key-pressed :path path}
             :on-mouse-dragged {::event/type ::on-mouse-dragged :path path}
             :on-mouse-pressed {::event/type ::on-mouse-pressed :path path}
             :on-mouse-released {::event/type ::on-mouse-released :path path}
             :on-width-changed {::event/type ::on-width-changed :path path}
             :on-height-changed {::event/type ::on-height-changed :path path}
             :on-scroll {::event/type ::on-scroll :path path}}

      popup
      (assoc :popup (assoc popup :fx/type view/view
                                 ::view/view ::popup/view
                                 :font (:font layout)
                                 :on-cancel {::event/type ::hide-popup :path path}
                                 :path (conj path :popup))))))

(defmethod view/view ::view [props]
  {:fx/type fx/ext-get-env
   :env [:reveal/css]
   :desc (assoc props :fx/type view-impl)})

(defn make [font]
  {:layout (layout/make {:font font :canvas-width 0.0 :canvas-height 0.0 :lines []})})

;; todo stateful components
;; - create initial state from some arguments
;; - allow props in addition to existing state
;; - state as effect/co-effect

#_{:tag :ret
   :val {{:a 1
          :x []} {1 0 0 1}
         :b 2}
   :prims [nil false true inc (keyword "weird\nkeyword")]
   :delay (delay 1)
   :future (future 50)
   :promise (promise)
   :file (.getCanonicalFile (java.io.File. "./file.css"))
   :vol (volatile! 1)
   :transient (transient [:a])
   :agent (agent #{::agent})
   :re #"[a-z]+"
   :atom (atom [String *ns*])
   :sym 'cljfx.fx.popup/lifecycle
   :var #'clojure.core/int?
   :e (RuntimeException. "beep")}

#_(fx/create-component
    {:fx/type :label
     :effect {:fx/type :drop-shadow}
     :graphic {:fx/type :v-box
               :children [{:fx/type :button :text "woop"}]}})

#_(prn :a 2)

#_(/ 1 0)

#_(with-meta {:a 1} {`clojure.core.protocols/nav vector})

#_(vec (map #(hash-map :n % % % :rand (rand)) (range 10000)))

#_(let [xf (reveal.stream/stream-xf (reveal.font/make "monospace" 15))
        x [{:tag :ret
            :val {{:a 1
                   :x []} {1 0 0 1}
                  :b 2}
            :prims [nil false true inc (keyword "weird\nkeyword")]
            :delay (delay 1)
            :future (future 50)
            :promise (promise)
            :file (.getCanonicalFile (java.io.File. "./deps.edn"))
            :vol (volatile! 1)
            :transient (transient [:a])
            :agent (agent #{::agent})
            :re #"[a-z]+"
            :atom (atom [String *ns*])
            :sym 'cljfx.fx.popup/lifecycle
            :var #'clojure.core/int?
            :e (RuntimeException. "beep")}]]
    (clj-async-profiler.core/profile
      {:return-file true
       :reverse? true}
      (criterium.core/quick-bench
        (into [] xf x))
      (flush)))
