(ns reveal.font
  (:require [cljfx.coerce :as fx.coerce])
  (:import [com.sun.javafx.tk Toolkit]
           [com.sun.javafx.font PGFont FontResource FontStrike]
           [com.sun.javafx.geom.transform BaseTransform]))

;; TODO java 8 support: in cljfx-css as well!

(set! *warn-on-reflection* true)

(set! *unchecked-math* :warn-on-boxed)

(deftype Font [^javafx.scene.text.Font font ^double line-height ^double ascent char-width-cache])

(def ^:private ^:const min-cached-char-width
  -42)

(def ^:private ^:const max-cached-char-width
  Double/MAX_VALUE)

(defn- make-char-width-cache [^FontStrike font-strike]
  (let [cache (double-array (inc (int Character/MAX_VALUE)) Double/MIN_VALUE)]
    (fn get-char-width [^Character character]
      (let [ch (unchecked-char character)
            i (unchecked-int ch)
            cached-width (aget cache i)]
        (if (= cached-width Double/MIN_VALUE)
          (let [width (.getCharAdvance font-strike ch)]
            (when (and (<= min-cached-char-width width)
                       (<= width max-cached-char-width))
              (aset cache i width))
            width)
          cached-width)))))

(defmacro ^:private if-class [class-name then else]
  `(try
     (Class/forName ^String ~class-name)
     ~then
     (catch ClassNotFoundException _#
       ~else)))

(def get-native-font
  (if-class "com.sun.javafx.scene.text.FontHelper"
    (let [meth (-> (Class/forName "com.sun.javafx.scene.text.FontHelper")
                   (.getDeclaredMethod "getNativeFont" (into-array Class [javafx.scene.text.Font])))]
      #(.invoke meth nil (into-array Object [%])))
    (let [meth (-> (Class/forName "javafx.scene.text.Font")
                   (.getDeclaredMethod "impl_getNativeFont" (into-array Class [])))]
      #(.invoke meth % (into-array Object [])))))

(defn make [family size]
  (let [^javafx.scene.text.Font font (fx.coerce/font {:family family :size size})
        metrics (.getFontMetrics (.getFontLoader (Toolkit/getToolkit)) font)
        strike (.getStrike ^PGFont (get-native-font font)
                           BaseTransform/IDENTITY_TRANSFORM
                           FontResource/AA_GREYSCALE)]
    (Font. font
           (Math/ceil (.getLineHeight metrics))
           (.getAscent metrics)
           (make-char-width-cache strike))))

(defn line-height ^double [^Font this]
  (.-line_height this))

(defn ascent ^double [^Font this]
  (.-ascent this))

(defn char-width ^double [^Font this char]
  ((.-char_width_cache this) char))

(defn jfx-font ^javafx.scene.text.Font [^Font this]
  (.-font this))
