(ns reveal.core
  (:require [clojure.core.server :as server]
            [cljfx.api :as fx]
            [clojure.edn :as edn]
            [reveal.font :as font]
            [clojure.main :as m]
            [cljfx.lifecycle :as fx.lifecycle]
            [reveal.stream :as stream]
            [reveal.output-panel :as output-panel]
            [reveal.event :as event]
            [reveal.writer-output-stream :as writer-output-stream]
            [reveal.parse :as parse]
            [reveal.view :as view]
            [reveal.style :as style])
  (:import [javafx.event Event]
           [java.util.concurrent Executors ThreadFactory ArrayBlockingQueue]
           [java.io PrintStream]
           [clojure.lang MultiFn]))

(defn- root-view-impl [{:keys [showing output reveal/css]}]
  {:fx/type :stage
   ;; todo ask if user want to quit repl (default) or exit jvm
   :on-close-request #(.consume ^Event %)
   :showing showing
   :width 400
   :height 500
   :on-focused-changed {::event/type ::output-panel/on-window-focus-changed
                        :path [:output]}
   :scene {:fx/type :scene
           :stylesheets [(:cljfx.css/url style/dark)]
           :root {:fx/type :v-box
                  :style {:-fx-background-color (::style/background-color css)}
                  :children [(assoc output :fx/type view/view
                                           ::view/view ::output-panel/view
                                           :v-box/vgrow :always
                                           :path [:output])]}}})

(defn- root-view [props]
  {:fx/type fx/ext-set-env
   :env {:reveal/css style/dark :reveal/font (:font props)}
   :desc {:fx/type fx/ext-get-env
          :env [:reveal/css]
          :desc (assoc props :fx/type root-view-impl)}})

(defn oneduce
  ([xf x]
   (oneduce xf identity x))
  ([xf f x]
   (let [rf (xf (completing #(f %2)))]
     (rf (rf nil x)))))

(def ^:private daemon-executor
  (let [*counter (atom 0)
        factory (reify ThreadFactory
                  (newThread [_ runnable]
                    (doto (Thread. runnable (str "reveal-agent-pool-" (swap! *counter inc)))
                      (.setDaemon true))))]
    (Executors/newCachedThreadPool factory)))

(defn- line-print-stream [line-fn]
  (let [sb (StringBuilder.)]
    (-> #(doseq [^char ch %]
           (if (= \newline ch)
             (let [str (.toString sb)]
               (.delete sb 0 (.length sb))
               (line-fn str))
             (.append sb ch)))
        (PrintWriter-on nil)
        (writer-output-stream/make)
        (PrintStream. true "UTF-8"))))

(defn- wrap-err-out [f output!]
  (fn []
    (let [out System/out
          err System/err]
      ;; todo don't steal, fork
      (System/setOut (line-print-stream #(-> % parse/system-out output!)))
      (System/setErr (line-print-stream #(-> % parse/system-err output!)))
      (f)
      (System/setOut out)
      (System/setErr err))))

(defn- wrap-renderer [f *state event-handler]
  (fn []
    (let [renderer (fx/create-renderer
                     :opts {:fx.opt/map-event-handler event-handler
                            :fx.opt/type->lifecycle #(or (fx/keyword->lifecycle %)
                                                         (fx/fn->lifecycle %)
                                                         (when (instance? MultiFn %)
                                                           fx.lifecycle/dynamic-fn->dynamic))}
                     :middleware (fx/wrap-map-desc #(root-view %)))]
      (fx/mount-renderer *state renderer)
      (f)
      (fx/unmount-renderer *state renderer))))

(defn- wrap-output-streaming [f out-queue xform to-fn]
  (fn []
    (let [output-thread (doto (Thread.
                                ^Runnable
                                (fn []
                                  (while (not (.isInterrupted (Thread/currentThread)))
                                    (try
                                      (let [x (.take out-queue)
                                            x (if (= ::stream-nil x) nil x)]
                                        (oneduce xform to-fn x))
                                      (catch InterruptedException _))))
                                "reveal-stream-thread")
                          (.setDaemon true))]
      (.start output-thread)
      (f)
      (.interrupt output-thread))))


(defn start [& {:keys [prepl streams]
                :or {prepl server/prepl
                     streams true}}]
  (let [out *out*
        err *err*
        font (font/make "Fantasque Sans Mono" 14.5)
        output-xf (comp (stream/stream-xf font)
                        (partition-all 128))
        out-queue (ArrayBlockingQueue. 1024)
        output! #(.put out-queue (if (nil? %) ::stream-nil %))
        *state (atom {:output (output-panel/make font)
                      :font font
                      :showing true})
        _ (def -state *state)
        event-handler (-> event/handle
                          (fx/wrap-co-effects
                            {:state (fx/make-deref-co-effect *state)})
                          (fx/wrap-effects
                            {:dispatch fx/dispatch-effect
                             :state (fx/make-reset-effect *state)})
                          (fx/wrap-async :error-handler #(binding [*out* err]
                                                           (prn %2))
                                         :fx/executor daemon-executor))
        repl (-> #(prepl *in* (fn [x]
                                (output! (parse/prepl-output x))
                                (binding [*out* out]
                                  (if (:exception x)
                                    (binding [*out* err]
                                      (println (m/ex-str (m/ex-triage (:val x))))
                                      (flush))
                                    (do
                                      (case (:tag x)
                                        :out (println (:val x))
                                        :err (binding [*out* err]
                                               (println (:val x)))
                                        (:tap :ret) (prn (:val x))
                                        (prn x))
                                      (flush)))
                                  (print (str (:ns x) "=> "))
                                  (flush))))
                 (wrap-output-streaming
                   out-queue
                   output-xf
                   #(event-handler {::event/type ::output-panel/add-lines
                                    :path [:output]
                                    :fx/event %}))
                 (wrap-renderer *state event-handler)
                 (cond-> streams (wrap-err-out output!)))
        v (str "Clojure " (clojure-version))]
    (output! (stream/as *clojure-version*
               (stream/raw-string v {:fill ::style/util-color})))
    (println v)
    (print (str (.name *ns*) "=> "))
    (flush)
    (repl)
    nil))

#_(require 'criterium.core)

#_(let [v [{:a 1 :b 2 :c (RuntimeException.)}]
        xf (stream/stream-xf (font/make "Fantasque Sans Mono" 15))
        rf (constantly nil)]
    (criterium.core/quick-bench
      (transduce xf rf nil v))
    (flush))

; protocol-based version
; Evaluation count : 504 in 6 samples of 84 calls.
;             Execution time mean : 1.246660 ms
;    Execution time std-deviation : 18.425994 µs
;   Execution time lower quantile : 1.225961 ms ( 2.5%)
;   Execution time upper quantile : 1.265027 ms (97.5%)
;                   Overhead used : 8.216258 ns

#_(-> -state deref :output :layout keys)

#_(start)

#_(/ 1 0)

#_(.start (Thread. #(prn 1)))

#_(do
    (require 'criterium.core)
    (criterium.core/quick-bench (+ 1 2)))

#_(range 10000)

#_(do :repl/quit)

#_(.printStackTrace (RuntimeException.))

(defn -main [& args]
  (apply start (map edn/read-string args)))

#_(stream/as 1 (stream/raw-string "aaa\tb" {:fill "#ccc"}))

#_(font/char-width (font/make "monospace" 14) \a)
#_(font/line-height (font/make "Fantasque Sans Mono" 15))
