(ns reveal.stream
  (:refer-clojure :exclude [newline])
  (:require [reveal.font :as font]))

(defn- =>
  ([] (fn [_ acc] acc))
  ([f] f)
  ([f g]
   (fn [rf acc]
     (g rf (f rf acc))))
  ([f g & fs]
   (reduce => (cons f (cons g fs)))))

(defn- op [op]
  (fn [rf acc]
    (rf acc op)))

(defn- with-value [x ann sf]
  (=> (op {:op ::push-value :value [x ann]})
      sf
      (op {:op ::pop-value})))

(def separator
  (op {:op ::separator}))

(defn- string [str style]
  (op {:op ::string
       :text str
       :style style}))

(def ^:private newline
  (op {:op ::newline}))

(defn- stream-dispatch [x]
  (or (:reveal/type (meta x))
      (class x)))

(defmulti parse stream-dispatch)

(defmulti annotate (fn [x ann sf]
                     (stream-dispatch x)))

(defmethod annotate :default [x ann sf]
  (with-value x ann sf))

(defmethod parse ::identity [x] x)

(defmethod annotate ::identity [_ _ sf]
  sf)

(defn stream
  "Streams value using default parsing"
  ([x]
   (stream x nil))
  ([x ann]
   (annotate x ann (parse x))))

(defn as
  "Streams value using custom sf"
  ([x sf]
   (as x nil sf))
  ([x ann sf]
   (with-meta (with-value x ann sf) {:reveal/type ::identity})))

(defn- flush-builder [^StringBuilder builder style]
  (fn [rf acc]
    (let [len (.length builder)]
      (if (pos? len)
        (let [ret ((string (.toString builder) style) rf acc)]
          (.delete builder 0 len)
          ret)
        acc))))

(defn- flush+util [builder style str]
  (=> (flush-builder builder style)
      (string str {:fill [:palette :util]})))

(defn- process-raw [^StringBuilder builder ^String str style]
  (fn [rf acc]
    (let [len (.length str)]
      (loop [i 0
             acc acc]
        (if (= i len)
          acc
          (let [ch (.charAt str i)]
            (recur
              (inc i)
              (case ch
                \newline ((=> (flush-builder builder style) newline) rf acc)
                \tab (do (.append builder "    ") acc)
                \return ((flush+util builder style "\\r") rf acc)
                \formfeed ((flush+util builder style "\\f") rf acc)
                \backspace ((flush+util builder style "\\b") rf acc)
                (do (.append builder ch) acc)))))))))

(defn- raw [str style]
  (let [builder (StringBuilder.)]
    (=> (process-raw builder str style)
        (flush-builder builder style))))

(defn- escaped [^String str style escape escape-style]
  (fn [rf acc]
    (let [len (.length str)
          builder (StringBuilder.)
          same-style (= style escape-style)]
      (loop [i 0
             acc acc]
        (if (= i len)
          ((flush-builder builder style) rf acc)
          (let [ch (.charAt str i)
                esc (escape ch)]
            (recur
              (inc i)
              (if esc
                (if same-style
                  ((process-raw builder esc style) rf acc)
                  ((=> (flush-builder builder style)
                       (process-raw builder esc escape-style)
                       (flush-builder builder escape-style))
                   rf acc))
                (case ch
                  \newline ((=> (flush-builder builder style) newline) rf acc)
                  \tab (do (.append builder "    ") acc)
                  \return ((flush+util builder style "\\r") rf acc)
                  \formfeed ((flush+util builder style "\\f") rf acc)
                  \backspace ((flush+util builder style "\\b") rf acc)
                  (do (.append builder ch) acc))))))))))

(defn- block [block-type sf]
  (=> (op {:op ::push-block :block block-type})
      sf
      (op {:op ::pop-block})))

(defn raw-string
  ([x]
   (raw-string x {}))
  ([x style]
   (block :paragraph (raw (str x) style))))

(defn escaped-string
  ([x escape]
   (escaped-string x {} escape {}))
  ([x style escape]
   (escaped-string x style escape style))
  ([x style escape escape-style]
   (block :paragraph (escaped (str x) style escape escape-style))))

(defn horizontal [& sfs]
  (block :horizontal (apply => sfs)))

(defn vertical [& sfs]
  (block :vertical (apply => sfs)))

(defn entries [m]
  (fn [rf acc]
    (reduce-kv
      (fn [acc k v]
        ((horizontal (stream k {:reveal.nav/val v})
                     (raw-string " ")
                     (stream v {:reveal.nav/key k})) rf acc))
      acc
      m)))

(defn- vertical-items [coll]
  (fn [rf acc]
    (let [*i (volatile! -1)]
      (reduce (fn [acc x]
                ((stream x {:reveal.nav/index (vswap! *i inc)}) rf acc))
              acc
              coll))))

(defn- horizontal-items [coll]
  (fn [rf acc]
    (let [*i (volatile! -1)]
      (transduce (interpose ::space)
                 (completing
                   (fn [acc x]
                     (if (= x ::space)
                       ((raw-string " ") rf acc)
                       ((stream x {:index (vswap! *i inc)}) rf acc))))
                 acc
                 coll))))

(defn items [coll]
  (if (some coll? coll)
    (block :vertical (vertical-items coll))
    (block :horizontal (horizontal-items coll))))

(defn sequential [xs]
  (block :vertical (vertical-items xs)))

(defmethod parse :default [x]
  (raw-string (pr-str x) {:fill [:palette :object]}))

(defn- parse-xf [rf]
  (fn
    ([] (rf))
    ([result] (rf result))
    ([result input] ((stream input) rf result))))

(defn- blank-segment [n font]
  {:text (apply str (repeat n \space))
   :width (* n (font/char-width font :regular \space))
   :style {}})

(defn- string-segment [string-op font]
  (let [kind (:font (:style string-op))]
    (-> string-op
        (dissoc :op)
        (assoc :width (transduce (map #(font/char-width font kind %)) + (:text string-op))))))

(defn- segment-length [segment]
  (count (:text segment)))

(defn- segments-length [segments]
  (transduce (map segment-length) + segments))

(defn- next-index [region]
  (+ (:index region 0)
     (segments-length (:segments region))))

(defn- add-segment [line values segment]
  (let [last-region (peek line)]
    (if (not= (:values last-region) values)
      (conj line {:values values :segments [segment] :index (next-index last-region)})
      (update-in line [(dec (count line)) :segments] conj segment))))

(defn- add-separator [line]
  (let [last-region (peek line)]
    (cond-> line
            last-region
            (conj {:values (:values last-region)
                   :segments []
                   :index (next-index last-region)}))))

(defn- line-length [line]
  (next-index (peek line)))

(defn- format-xf [font]
  (fn [rf]
    (let [*state (volatile! {:line [] :values [] :blocks []})]
      (fn
        ([] (rf))
        ([result] (rf result))
        ([result input]
         (let [state @*state]
           (case (:op input)
             ::push-value
             (do (vswap! *state update :values conj (:value input))
                 result)

             ::pop-value
             (do (vswap! *state update :values pop)
                 result)

             ::push-block
             (let [blocks (:blocks state)
                   block (peek blocks)]
               (case (:block block)
                 :vertical
                 (if (:had-sub-blocks block)
                   (do (vreset!
                         *state
                         (assoc state
                           :line (-> []
                                     (add-segment (:values state) (blank-segment (:indent block) font))
                                     (add-separator))
                           :blocks (conj blocks {:block (:block input)
                                                 :indent (:indent block)})))
                       (rf result (:line state)))
                   (do (vswap! *state assoc :blocks
                               (-> blocks
                                   (assoc-in [(dec (count blocks)) :had-sub-blocks] true)
                                   (conj {:block (:block input)
                                          :indent (:indent block)})))
                       result))

                 :horizontal
                 (do (vswap! *state update :blocks conj
                             {:block (:block input)
                              :indent (line-length (:line state))})
                     result)

                 nil
                 (do (vswap! *state update :blocks conj
                             {:block (:block input)
                              :indent 0})
                     result)))

             ::pop-block
             (let [blocks (:blocks state)]
               (vswap! *state update :blocks pop)
               (if (= 1 (count blocks))
                 (do (vreset! *state (-> state
                                         (assoc :blocks (pop blocks))
                                         (assoc :line [])))
                     (rf result (:line state)))
                 (do (vreset! *state (assoc state :blocks (pop blocks)))
                     result)))

             ::separator
             (do (vswap! *state update :line add-separator)
                 result)

             ::string
             (do (vswap! *state update :line add-segment (:values state) (string-segment input font))
                 result)

             ::newline
             (let [block (peek (:blocks state))]
               (do (vswap! *state assoc :line (add-segment [] (:values state) (blank-segment (:indent block) font)))
                   (rf result (:line state)))))))))))

(defn stream-xf [font]
  (comp parse-xf (format-xf font)))

#_ (clojure.pprint/pprint (into [] (stream-xf (reveal.font/make "monospace" 15)) [(reveal.parse/prepl-output {:tag :out :val "a"})]))

#_(into [] (stream-xf (reveal.font/make "monospace" 15)) [(ex-info "" {})])
