(ns reveal.parse
  (:require [reveal.stream :as stream]
            [clojure.main :as m])
  (:import [clojure.lang Keyword Symbol IPersistentMap IPersistentVector IPersistentSet Fn IPersistentList ISeq MultiFn
                         IRef Var Volatile Namespace IRecord Delay IBlockingDeref TaggedLiteral Reduced]
           [java.util.regex Pattern]
           [java.io File]
           [java.net URL]))

(set! *warn-on-reflection* true)

(defn- identity-hash-code [x]
  (let [hash (System/identityHashCode x)]
    (stream/as hash
      (stream/raw-string (format "0x%x" hash) {:fill [:palette :scalar]}))))

(defmethod stream/parse nil [x]
  (stream/raw-string (pr-str x) {:fill [:palette :scalar]}))

(defmethod stream/parse Boolean [x]
  (stream/raw-string (pr-str x) {:fill [:palette :scalar]}))

(defmethod stream/parse String [s]
  (stream/raw-string (pr-str s) {:fill [:palette :string]}))

(defmethod stream/parse Character [ch]
  (stream/raw-string (pr-str ch) {:fill [:palette :string]}))

(defmethod stream/parse Keyword [k]
  (stream/ensure-simple-string (str k) {:fill [:palette :keyword]}))

(defmethod stream/parse Symbol [sym]
  (stream/ensure-simple-string (str sym) {:fill [:palette :symbol]}))

(defmethod stream/parse Number [n]
  (stream/ensure-simple-string (str n) {:fill [:palette :scalar]}))

(defmethod stream/parse Float [n]
  (stream/raw-string
    (cond
      (= Float/POSITIVE_INFINITY n) "##Inf"
      (= Float/NEGATIVE_INFINITY n) "##-Inf"
      (Float/isNaN n) "##NaN"
      :else (str n))
    {:fill [:palette :scalar]}))

(defmethod stream/parse Double [n]
  (stream/raw-string
    (cond
      (= Double/POSITIVE_INFINITY n) "##Inf"
      (= Double/NEGATIVE_INFINITY n) "##-Inf"
      (Double/isNaN n) "##NaN"
      :else (str n))
    {:fill [:palette :scalar]}))

(defmethod stream/parse IPersistentMap [m]
  (stream/horizontal
    (stream/raw-string "{" {:fill [:palette :object]})
    (stream/vertical (stream/entries m))
    (stream/raw-string "}" {:fill [:palette :object]})))

(defmethod stream/parse IRecord [m]
  (stream/horizontal
    (stream/raw-string (str "#" (.getName (class m)) "{") {:fill [:palette :object]})
    (stream/vertical (stream/entries m))
    (stream/raw-string "}" {:fill [:palette :object]})))

(prefer-method stream/parse IRecord IPersistentMap)
(prefer-method stream/parse ISeq IPersistentList)

(defmethod stream/parse IPersistentVector [v]
  (stream/horizontal
    (stream/raw-string "[" {:fill [:palette :object]})
    (stream/items v)
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse IPersistentList [v]
  (stream/horizontal
    (stream/raw-string "(" {:fill [:palette :object]})
    (stream/items v)
    (stream/raw-string ")" {:fill [:palette :object]})))

(defmethod stream/parse ISeq [xs]
  (stream/horizontal
    (stream/raw-string "(" {:fill [:palette :object]})
    (stream/sequential xs)
    (stream/raw-string ")" {:fill [:palette :object]})))

(defmethod stream/parse IPersistentSet [s]
  (stream/horizontal
    (stream/raw-string "#{" {:fill [:palette :object]})
    (stream/items s)
    (stream/raw-string "}" {:fill [:palette :object]})))

(defmethod stream/parse Throwable [t]
  (stream/horizontal
    (stream/raw-string "#error" {:fill [:palette :object]})
    (stream/stream (Throwable->map t))))

(defmethod stream/parse MultiFn [^MultiFn f]
  (stream/horizontal
    (stream/raw-string "#multi-fn[" {:fill [:palette :object]})
    (identity-hash-code f)
    (stream/raw-string " ")
    (stream/stream (.-dispatchFn f))
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse Fn [f]
  (stream/ensure-simple-string
    (Compiler/demunge (.getName (class f)))
    {:fill [:palette :object]}))

(defmethod stream/parse Pattern [re]
  (stream/horizontal
    (stream/raw-string "#" {:fill [:palette :object]})
    (stream/ensure-simple-string (str \" re \") {:fill [:palette :string]})))

(defmethod stream/parse Var [var]
  (stream/raw-string (pr-str var) {:fill [:palette :object]}))

(defmethod stream/parse Namespace [^Namespace ns]
  (stream/raw-string (name (.getName ns)) {:fill [:palette :object]}))

(defmethod stream/parse Class [^Class class]
  (stream/raw-string (.getName class) {:fill [:palette :object]}))

(defmethod stream/parse Enum [^Enum enum]
  (stream/raw-string (str (.getName (.getDeclaringClass enum)) "/" (.name enum))
                     {:fill [:palette :scalar]}))

(defmethod stream/parse IRef [*ref]
  (stream/horizontal
    (stream/raw-string (str "#" (.toLowerCase (.getSimpleName (class *ref))) "[") {:fill [:palette :object]})
    (identity-hash-code *ref)
    (stream/raw-string " ")
    (stream/stream @*ref)
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse File [file]
  (stream/horizontal
    (stream/raw-string "#file[" {:fill [:palette :object]})
    stream/separator
    (stream/ensure-simple-string (str file) {:fill [:palette :string]})
    stream/separator
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse Delay [*delay]
  (stream/horizontal
    (stream/raw-string "#delay[" {:fill [:palette :object]})
    (identity-hash-code *delay)
    (stream/raw-string " ")
    (if (realized? *delay)
      (stream/stream @*delay)
      (stream/raw-string "..." {:fill [:palette :util]}))
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse Reduced [*reduced]
  (stream/horizontal
    (stream/raw-string "#reduced[" {:fill [:palette :object]})
    (stream/stream @*reduced)
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse IBlockingDeref [*blocking-deref]
  (let [class-name (.getName (class *blocking-deref))]
    (cond
      (.startsWith class-name "clojure.core$promise$reify")
      (stream/horizontal
        (stream/raw-string "#promise[" {:fill [:palette :object]})
        (identity-hash-code *blocking-deref)
        (stream/raw-string " ")
        (if (realized? *blocking-deref)
          (stream/stream @*blocking-deref)
          (stream/raw-string "..." {:fill [:palette :util]}))
        (stream/raw-string "]" {:fill [:palette :object]}))

      (.startsWith class-name "clojure.core$future_call$reify")
      (stream/horizontal
        (stream/raw-string "#future[" {:fill [:palette :object]})
        (identity-hash-code *blocking-deref)
        (stream/raw-string " ")
        (if (realized? *blocking-deref)
          (stream/stream @*blocking-deref)
          (stream/raw-string "..." {:fill [:palette :util]}))
        (stream/raw-string "]" {:fill [:palette :object]}))

      :else
      (stream/multiline (pr-str *blocking-deref) {:fill [:palette :object]}))))

(defmethod stream/parse Volatile [*ref]
  (stream/horizontal
    (stream/raw-string "#volatile[" {:fill [:palette :object]})
    (identity-hash-code *ref)
    (stream/raw-string " ")
    (stream/stream @*ref)
    (stream/raw-string "]" {:fill [:palette :object]})))

(defmethod stream/parse TaggedLiteral [x]
  (stream/horizontal
    (stream/raw-string "#" {:fill [:palette :object]})
    (stream/stream (:tag x))
    (stream/raw-string " ")
    (stream/stream (:form x))))

#_(defmethod stream/parse URL [x]
    (stream/horizontal
      (stream/raw-string "#url" {:fill [:palette :object]})))

#_(str (java.net.URL. "ftp://1.1.1.1/file.zip?q= asdh kahsd #\nwat"))

#_(clojure.java.io/file "asd")
#_(clojure.java.io/resource "META-INF/services/java.net.spi.URLStreamHandlerProvider")
#_(clojure.java.io/resource "META-INF/services/java.net.spi.URLStreamHandlerProvider")

(defn prepl-output [x]
  (stream/as x
    (if (:exception x)
      (cond-> (stream/multiline (-> x :val m/ex-triage m/ex-str) {:fill [:palette :error]})
              (:form x)
              (as-> err-output
                    (stream/vertical
                      (stream/multiline (:form x) {:fill [:palette :util]})
                      err-output)))
      (case (:tag x)
        :ret (stream/vertical
               (stream/multiline (:form x) {:fill [:palette :util]})
               (stream/horizontal
                 (stream/raw-string "=>" {:fill [:palette :util]})
                 stream/separator
                 (stream/raw-string " " {:fill [:palette :util]})
                 (stream/stream (:val x))))
        :out (stream/multiline (:val x) {:fill [:palette :string]})
        :err (stream/multiline (:val x) {:fill [:palette :error]})
        :tap (stream/horizontal
               (stream/raw-string "tap>" {:fill [:palette :util]})
               stream/separator
               (stream/raw-string " " {:fill [:palette :util]})
               (stream/stream (:val x)))
        (stream/parse x)))))

(defn action-output [action ret]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill [:palette :util]})
        (stream/stream (:id action)))
      (stream/horizontal
        (stream/raw-string "$> " {:fill [:palette :util]})
        (stream/stream ret)))))

(defn action-error-output [action ex]
  (stream/as action
    (stream/vertical
      (stream/horizontal
        (stream/raw-string "$action " {:fill [:palette :util]})
        (stream/stream (:id action)))
      (stream/stream ex nil))))

(defn system-out [line]
  (stream/as line
    (stream/raw-string line {:fill [:palette :string]})))

(defn system-err [line]
  (stream/as line
    (stream/raw-string line {:fill [:palette :error]})))
