(ns ksuid.core
  (:require [clojure.string]
            [ksuid.base62 :as base62]))

(comment
  "Description from https://github.com/segmentio/ksuid#how-do-ksuids-work:
   Binary KSUIDs are 20-bytes: a 32-bit unsigned integer UTC timestamp and a 128-bit randomly generated payload. 
   The timestamp uses big-endian encoding, to support lexicographic sorting. The timestamp epoch is adjusted to May 13th, 2014, providing over 100 years of life. 
   The payload is generated by a cryptographically-strong pseudorandom number generator.")

;; KSUID's epoch starts more recently so that the 32-bit number space gives a
;; significantly higher useful lifetime of around 136 years from March 2017.
;; This number (14e8) was picked to be easy to remember.
(def epoch-stamp 1400000000)

(def timestamp-byte-length 4)
(def payload-byte-length 16)
(def byte-length (+ timestamp-byte-length payload-byte-length))
;; KSUID always have a length of 27
(def encoding-length 27)

(defn- leftpad
  [s length char]
  (let [length-to-pad (- length (count s))]
    (str (clojure.string/join (repeat length-to-pad char)) s)))

(defprotocol KSUIDProtocol
  (to-string [this])
  (time-instant [this]))

(defrecord KSUID [timestamp payload bytes]
  KSUIDProtocol
  (to-string [ksuid] (-> (base62/encode-bytes (:bytes ksuid))
                         (leftpad encoding-length "0")))
  (time-instant [ksuid] (-> (+ epoch-stamp (:timestamp ksuid))
                            (java.time.Instant/ofEpochSecond))))

(defn- to-corrected-timestamp [timestamp]
  (- timestamp epoch-stamp))

(defn- generate-random-bytes
  "Generate a cryptographically strong random byte-array"
  [size]
  (let [bytes (byte-array size)]
    (.nextBytes (java.security.SecureRandom.) bytes)
    bytes))

(defn- new-bytes
  "Create a new KSUID in byte-array. 
   Use this if you only care about the underlying byte-array,
   otherwise use ksuid()."
  [timestamp payload]
  (-> (java.nio.ByteBuffer/allocate byte-length)
      (.putInt timestamp)
      (.put payload)
      (.array)))

(defn from-parts
  [time-instant payload]
  (let [timestamp (to-corrected-timestamp (.getEpochSecond time-instant))
        bytes (new-bytes timestamp payload)]
    (->KSUID timestamp payload bytes)))

(defn new-random-with-time
  [time-instant]
  (from-parts time-instant (generate-random-bytes 16)))

(defn new-random
  "Create a new KSUID."
  [] (new-random-with-time (java.time.Instant/now)))

(defn from-bytes
  [bytes]
  (let [timestamp (->> (take timestamp-byte-length bytes)
                       (byte-array)
                       (BigInteger. 1))
        payload (byte-array (drop timestamp-byte-length bytes))]
    (->KSUID timestamp payload bytes)))

(defn from-string
  [s]
  (from-bytes (base62/decode-bytes s)))

(defn valid?
  "Check if a string is a valid KSUID."
  [ksuid]
  (cond
    (not= (count ksuid) encoding-length) false
    (not (base62/base62? ksuid)) false
    :else true))


(comment
  (new-random)
  (new-random-with-time (java.time.Instant/ofEpochSecond 1400000001))
  (to-string (new-random))
  (time-instant (new-random))
  (from-parts (java.time.Instant/ofEpochSecond 1400000001) (byte-array [19, -96, 65, 35, -33, 100, -105, -61, 51, -121, 19, -22, -98, -109, -92, 29]))
  (from-string  "2EBPQok6t9M6pFH8watHYNfs390")
  (from-bytes (byte-array [15, -98, -49, 70, 113, 9, -97, 82, -14, 63, -22, -16, -41, -69, 58, -20, -53, -94, 42, 58])))
