(ns ksuid.core
  (:require [clojure.string]
            [ksuid.base62 :as base62]))

(comment
  "Description from https://github.com/segmentio/ksuid#how-do-ksuids-work:
   Binary KSUIDs are 20-bytes: a 32-bit unsigned integer UTC timestamp and a 128-bit randomly generated payload. 
   The timestamp uses big-endian encoding, to support lexicographic sorting. The timestamp epoch is adjusted to May 13th, 2014, providing over 100 years of life. 
   The payload is generated by a cryptographically-strong pseudorandom number generator.")

(defrecord KSUID [timestamp payload bytes])

;; KSUID's epoch starts more recently so that the 32-bit number space gives a
;; significantly higher useful lifetime of around 136 years from March 2017.
;; This number (14e8) was picked to be easy to remember.
(def epoch-stamp 1400000000)

(def timestamp-byte-length 4)
(def payload-byte-length 16)
(def byte-length (+ timestamp-byte-length payload-byte-length))
;; KSUID always have a length of 27
(def encoding-length 27)

(defn- to-corrected-timestamp [timestamp]
  (- timestamp epoch-stamp))

(defn- generate-random-bytes
  "Generate a cryptographically strong random byte-array"
  [size]
  (let [bytes (byte-array size)]
    (.nextBytes (java.security.SecureRandom.) bytes)
    bytes))

(defn- new-bytes
  "Create a new KSUID in byte-array. 
   Use this if you only care about the underlying byte-array,
   otherwise use ksuid()."
  [timestamp payload]
  (-> (java.nio.ByteBuffer/allocate byte-length)
      (.putInt timestamp)
      (.put payload)
      (.array)))

(defn new-random-with-time
  [time-instant]
  (let [timestamp (to-corrected-timestamp (.getEpochSecond time-instant))
        payload (generate-random-bytes 16)]
    (let [bytes (new-bytes timestamp payload)]
      (->KSUID timestamp payload bytes))))

(defn new-random
  "Create a new KSUID."
  [] (new-random-with-time (java.time.Instant/now)))

(defn from-parts
  [time-instant payload]
  (let [timestamp (to-corrected-timestamp (.getEpochSecond time-instant))
        bytes (new-bytes timestamp payload)]
    (->KSUID timestamp payload bytes)))

(defn from-bytes
  [bytes]
  (throw (ex-info "Not Implemented" {})))

(defn from-string
  [s]
  (throw (ex-info "Not Implemented" {})))

(defn valid?
  "Check if a string is a valid KSUID."
  [ksuid]
  ;; TODO: (regex) match only base62 characters)
  (cond
    (not= (count ksuid) encoding-length) false
    (not (base62/base62? ksuid)) false
    :else true))

(defprotocol KSUIDP
  (string [this])
  (time-instant [this]))

(defn- leftpad
  [s length char]
  (let [length-to-pad (- length (count s))]
    (str (clojure.string/join (repeat length-to-pad char)) s)))

(extend-type KSUID
  KSUIDP
  (string [ksuid] (-> (base62/encode-bytes (:bytes ksuid))
                      (leftpad encoding-length "0")))
  (time-instant [ksuid] (-> (+ epoch-stamp (:timestamp ksuid))
                            (java.time.Instant/ofEpochSecond))))

(comment
  (new-random)
  (new-random-with-time (java.time.Instant/ofEpochSecond 1400000001))
  (string (new-random))
  (time-instant (new-random))
  (from-parts (java.time.Instant/ofEpochSecond 1400000001) (byte-array [19, -96, 65, 35, -33, 100, -105, -61, 51, -121, 19, -22, -98, -109, -92, 29])))
