(ns atlas.registry
  "Semantic kernel providing compound-identity algebra.
   Each identity is a set of qualified keywords representing composed meaning.
   All operations are deterministic, data-oriented, and side-effect free,
   except for explicit registry mutation helpers."
  (:require [clojure.set :as set]
            [clojure.string :as str]
            [atlas.query :as query]))

;; =============================================================================
;; Registry
;; =============================================================================

(def ^:dynamic env-id #{})

(def registry
  (atom {}))

;; =============================================================================
;; Validation & Registry
;; =============================================================================

(defn valid?
  "Returns true if `id` is a valid compound identity.
   A valid identity is a set of at least 2 qualified keywords."
  ([id]
   (valid? id false))
  ([id verbose?]
   (cond
     (not (set? id)) (if verbose? [:error "not-a-set"] false)
;;     (< (count id) 2) (if verbose? [:error "too-few-elements"] false)
     (not-every? qualified-keyword? id) (if verbose? [:error "unqualified-keywords"] false)
     :else true)))

(defn fetch
  "Fetch the value associated with a compound identity."
  [identity]
  (get @registry identity))

(defn exists?
  "Check if a compound identity exists in the registry and whether its value matches `value`.
   Returns {:exists bool :matches bool}."
  [compound-identity value]
  (let [found (get @registry compound-identity)]
    {:exists (contains? @registry compound-identity)
     :matches (= found value)}))

(defn remove*
  "Remove a compound identity from the registry."
  [compound-identity]
  (swap! registry dissoc compound-identity))

(defn- find-by-dev-id
  "Find the compound identity that has the given dev-id.
   Returns [compound-id value] or nil if not found.
   PRIVATE: Used internally by register! for dev-id uniqueness checking."
  [dev-id]
  (when dev-id
    (first (filter (fn [[_k v]] (= (:dev/id v) dev-id))
                   @registry))))

(defn generate-dev-id
  "Generate a deterministic dev-id from a compound identity (aspect set).
   Sorts aspects alphabetically and creates a keyword in the 'auto' namespace."
  [aspect-set]
  (let [sorted-aspects (sort aspect-set)
        aspect-names (map (fn [kw]
                            (str (namespace kw) "--" (name kw)))
                          sorted-aspects)
        joined (str/join "--" aspect-names)]
    (keyword "auto" joined)))

(defn register!
  "Register a new compound identity and its associated value.
   Asserts validity before inserting.
   Ensures compound ID and dev-id uniqueness.
   Auto-generates dev-id if not provided.

   Two arities:
   - [id value]: Register with auto-generated or provided dev-id
   - [dev-id id value]: Register with explicit dev-id"
  ([id value]
   (assert (valid? id))
   ;; Generate dev-id if needed and update value if it's a map
   (let [dev-id (if (and (associative? value) (:dev/id value))
                  (:dev/id value)
                  (generate-dev-id id))
         value-with-id (if (associative? value)
                         (assoc value :dev/id dev-id)
                         value)]  ; Keep non-map values as-is
     ;; Check for duplicate compound ID
     (when (contains? @registry id)
       (throw (ex-info "Compound identity already exists in registry"
                       {:compound-id id
                        :existing-value (get @registry id)
                        :attempted-value value-with-id})))
     ;; Check for duplicate dev-id
     (when-let [[existing-id existing-value] (find-by-dev-id dev-id)]
       (when (not= existing-id id)
         (throw (ex-info "dev-id already exists in registry"
                         {:dev-id dev-id
                          :existing-compound-id existing-id
                          :existing-value existing-value
                          :attempted-compound-id id
                          :attempted-value value-with-id}))))
     (swap! registry assoc id value-with-id)
     id))
  ([dev-id id value]
   (assert (valid? id) (format "...%s..." id))
   ;; Check for duplicate compound ID
   (when (contains? @registry id)
     (throw (ex-info "Compound identity already exists in registry"
                     {:compound-id id
                      :existing-value (get @registry id)
                      :attempted-value (assoc value :dev/id dev-id)})))
   ;; Check for duplicate dev-id
   (when-let [[existing-id existing-value] (find-by-dev-id dev-id)]
     (when (not= existing-id id)
       (throw (ex-info "dev-id already exists in registry"
                       {:dev-id dev-id
                        :existing-compound-id existing-id
                        :existing-value existing-value
                        :attempted-compound-id id
                        :attempted-value (assoc value :dev/id dev-id)}))))
   (swap! registry assoc id (assoc value :dev/id dev-id))
   id))

;; =============================================================================
;; Query Functions Moved to atlas.query
;; =============================================================================
;;
;; The following query functions have been moved to atlas.query namespace:
;; - all-identities → atlas.query/all-identities
;; - query → atlas.query/query-superset
;; - find-with → atlas.query/find-by-aspect
;; - find-exact → atlas.query/find-exact
;; - related-to → atlas.query/related-aspects
;; - semantic-neighbors → atlas.query/semantic-similarity
;; - query-algebra → atlas.query/query-algebra
;; - where → atlas.query/where
;; - aspect-frequency → atlas.query/aspect-frequency
;; - identity-stats → atlas.query/identity-stats
;;
;; Use atlas.query directly for all query operations.
;; =============================================================================

;; =============================================================================
;; Analytical Utilities (Domain-Specific to Registry)
;; =============================================================================

(defn clusters
  "Group identities by primary namespace, returning a map {:ns [identities]}."
  []
  (reduce (fn [acc id-set]
            (reduce (fn [inner aspect]
                      (let [ns-part (namespace aspect)]
                        (update inner (keyword ns-part)
                                (fnil conj #{}) id-set)))
                    acc id-set))
          {}
          (query/all-identities @registry)))

(defn correlation-matrix
  "Return normalized correlation matrix of aspect co-occurrence (as 0–1 floats)."
  []
  (let [aspects (keys (query/aspect-frequency @registry))
        all-ids (query/all-identities @registry)
        total (max 1 (count all-ids))]
    (into {}
          (for [a1 aspects]
            [a1 (into {}
                      (for [a2 aspects]
                        [a2 (/ (count (filter #(and (contains? % a1)
                                                    (contains? % a2))
                                              all-ids))
                               total)]))]))))

;; =============================================================================
;; Heuristics & Diagnostics
;; =============================================================================

(defn missing-aspects
  "Suggest aspects potentially missing from an identity based on correlation."
  [identity-set]
  (let [similar (query/semantic-similarity @registry identity-set 0.0)
        all-aspects (->> similar
                         (mapcat :identity)
                         (frequencies)
                         (sort-by val >))]
    (->> all-aspects
         (remove (fn [[aspect _]] (contains? identity-set aspect)))
         (take 5)
         (mapv (fn [[aspect freq]]
                 {:aspect aspect
                  :correlation (/ freq (count similar))})))))

(defn find-anomalies
  "Return list of identities that violate expected semantic constraints."
  []
  (let [all-ids (query/all-identities @registry)]
    (->> all-ids
         (filter (fn [id]
                   (or (and (contains? id :sync/operation)
                            (contains? id :async/operation))
                       (and (contains? id :api/endpoint)
                            (not (contains? id :auth/required)))))))))

;; =============================================================================
;; Visualization
;; =============================================================================

(defn to-graphviz
  "Return a Graphviz DOT string representing compound-identity relationships."
  []
  (let [edges (for [id (query/all-identities @registry)
                    aspect id]
                [aspect id])]
    (str "digraph CompoundIdentity {\n"
         "  rankdir=LR;\n"
         "  node [shape=box];\n"
         (str/join "\n"
                   (for [[aspect id] edges]
                     (format "  \"%s\" -> \"%s\";"
                             (pr-str aspect)
                             (pr-str id))))
         "\n}")))

(defn to-mermaid
  "Return a Mermaid diagram (graph TD) for namespace-based clusters."
  []
  (let [clusters (clusters)]
    (str "graph TD\n"
         (str/join "\n"
                   (for [[ns-key ids] clusters]
                     (str "  subgraph " (name ns-key) "\n"
                          (str/join "\n"
                                    (for [id ids]
                                      (format "    %s[\"%s\"]"
                                              (hash id)
                                              (str/join ", " (map name id)))))
                          "\n  end"))))))

;; =============================================================================
;; Summary
;; =============================================================================

(defn summary
  "Return data summary of the current identity registry:
   {:total :unique-aspects :namespaces :top-aspects :largest :anomalies}"
  []
  (let [ids (query/all-identities @registry)
        aspects (query/aspect-frequency @registry)
        clusters (clusters)]
    {:total (count ids)
     :unique-aspects (count aspects)
     :namespaces (keys clusters)
     :top-aspects (take 5 aspects)
     :largest (take 3 (sort-by count > ids))
     :anomalies (find-anomalies)}))
