(ns atlas.invariant.datalog
  "DATALOG-BASED INVARIANT CHECKING"
  (:require [atlas.graph.datalog :as graph.datalog]))

(defn check-invariant-datalog
  "Check an invariant using Datalog queries.
   This is an alternative implementation to the DSL-based checker."
  [db invariant-fn invariant-id level]
  (let [violations (invariant-fn db)]
    (when (seq violations)
      {:invariant invariant-id
       :level level
       :violations (vec violations)})))

(defn datalog-invariants
  "Standard invariants implemented using Datalog queries."
  [db]
  (remove nil?
          [(check-invariant-datalog
            db
            (fn [db] (map (fn [[ctx consumer]] {:entity consumer :missing-key ctx})
                          (graph.datalog/query-missing-producers db)))
            :invariant/no-missing-producers
            :error)

           (check-invariant-datalog
            db
            (fn [db] (map (fn [[resp producer]] {:entity producer :orphan-key resp})
                          (graph.datalog/query-orphan-outputs db)))
            :invariant/no-orphan-outputs
            :warning)

           (check-invariant-datalog
            db
            (fn [db] (map (fn [[dev missing]] {:entity dev :missing-dep missing})
                          (graph.datalog/query-missing-dependencies db)))
            :invariant/deps-exist
            :error)

           (check-invariant-datalog
            db
            (fn [db] (when-let [cycle (graph.datalog/query-acyclic-deps db)]
                       [{:cycle cycle}]))
            :invariant/no-circular-deps
            :error)

           (check-invariant-datalog
            db
            (fn [db] (map (fn [fn-id] {:entity fn-id})
                          (graph.datalog/query-unreachable-functions db)))
            :invariant/all-fns-reachable
            :warning)

           ;; Protocol invariants
           (check-invariant-datalog
            db
            (fn [db] (map (fn [protocol-aspect] {:missing-protocol protocol-aspect})
                          (graph.datalog/query-undefined-protocols db)))
            :invariant/protocol-exists
            :error)]))

(defn check-all-datalog
  "Check all invariants using Datalog backend."
  []
  (let [db (graph.datalog/create-db)
        violations (datalog-invariants db)]
    {:violations (vec violations)
     :errors (filterv #(= :error (:level %)) violations)
     :warnings (filterv #(= :warning (:level %)) violations)
     :valid? (empty? (filter #(= :error (:level %)) violations))}))

