 (ns atlas.invariant
  "Fixed invariants that understand dataflow markers and terminal outputs."
  (:require [atlas.registry :as cid]
            [atlas.graph :as rt]
            [clojure.set :as set]))

;; =============================================================================
;; HELPERS
;; =============================================================================

(defn all-dev-ids []
  (map #(:dev/id (second %)) @cid/registry))

(defn all-context-keys []
  (->> @cid/registry vals (mapcat :semantic-namespace/context) (remove nil?) set))

(defn all-response-keys []
  (->> @cid/registry vals (mapcat :semantic-namespace/response) (remove nil?) set))

(defn endpoint-context-keys []
  "Keys that come from outside (endpoint inputs)."
  (->> @cid/registry
       (filter (fn [[id _]] (contains? id :semantic-namespace/endpoint)))
       (mapcat (fn [[_ v]] (:semantic-namespace/context v)))
       set))

(defn endpoint-response-keys []
  "Keys that go to the client (endpoint outputs) - terminal by design."
  (->> @cid/registry
       (filter (fn [[id _]] (contains? id :semantic-namespace/endpoint)))
       (mapcat (fn [[_ v]] (:semantic-namespace/response v)))
       set))

(defn display-output-keys []
  "Keys marked as :dataflow/display-output - terminal by design."
  (->> @cid/registry
       (filter (fn [[id _]] (contains? id :dataflow/display-output)))
       (map (fn [[id _]]
              ;; The key IS the dev-id for data-keys
              (:dev/id (cid/fetch id))))
       (remove nil?)
       set))

(defn external-input-keys []
  "Keys marked as :dataflow/external-input - no internal producer expected."
  (->> @cid/registry
       (filter (fn [[id _]] (contains? id :dataflow/external-input)))
       (map (fn [[id _]] (:dev/id (cid/fetch id))))
       (remove nil?)
       set))

;; =============================================================================
;; FIXED DATA FLOW AXIOMS
;; =============================================================================

(defn invariant-context-satisfiable
  "Every context key must be either:
   - An endpoint input (comes from client)
   - Marked as :dataflow/external-input
   - Produced by some function"
  []
  (let [endpoint-inputs (endpoint-context-keys)
        external-inputs (external-input-keys)
        produced (all-response-keys)
        available (set/union endpoint-inputs external-inputs produced)
        all-needed (all-context-keys)
        unsatisfied (set/difference all-needed available)]
    (when (seq unsatisfied)
      {:invariant :context-satisfiable
       :violation :unsatisfied-context
       :missing unsatisfied
       :severity :error
       :message (str "These context keys are needed but never produced: " unsatisfied)})))

(defn invariant-no-orphan-responses
  "Response keys should be consumed OR be terminal outputs.
   
   A key is NOT orphan if any of:
   - Consumed by another function's context
   - Appears in an endpoint response (goes to client)
   - Marked as :dataflow/display-output (UI-bound)
   
   This catches truly dead code while allowing legitimate terminal outputs."
  []
  (let [produced (all-response-keys)
        consumed (all-context-keys)
        endpoint-outputs (endpoint-response-keys)
        display-outputs (display-output-keys)
        ;; Terminal = goes somewhere outside the internal dataflow
        terminal (set/union endpoint-outputs display-outputs)
        ;; Orphan = produced but neither consumed internally nor terminal
        orphans (-> produced
                    (set/difference consumed)
                    (set/difference terminal))]
    (when (seq orphans)
      {:invariant :no-orphan-responses
       :violation :orphan-outputs
       :orphans orphans
       :severity :warning
       :message (str "These response keys are produced but never consumed or displayed: " orphans)})))

(defn invariant-no-orphan-responses-strict
  "Strict version: only allows consumed keys, not terminal.
   Use for internal service layers where everything should chain."
  []
  (let [;; Only look at non-endpoint functions
        internal-fns (->> @cid/registry
                          (filter (fn [[id _]]
                                    (and (contains? id :semantic-namespace/function)
                                         (not (contains? id :semantic-namespace/endpoint)))))
                          (map second))
        produced (->> internal-fns (mapcat :semantic-namespace/response) (remove nil?) set)
        consumed (all-context-keys)
        orphans (set/difference produced consumed)]
    (when (seq orphans)
      {:invariant :no-orphan-responses-strict
       :violation :orphan-internal-outputs
       :orphans orphans
       :severity :warning
       :message (str "Internal function outputs not consumed: " orphans)})))

;; =============================================================================
;; DEPENDENCY AXIOMS (unchanged logic, better messages)
;; =============================================================================

(defn invariant-deps-exist
  "All :semantic-namespace/deps must reference existing dev-ids."
  []
  (let [all-ids (set (all-dev-ids))
        violations (for [[_ v] @cid/registry
                         :let [dev-id (:dev/id v)
                               deps (:semantic-namespace/deps v)]
                         :when deps
                         :let [missing (set/difference deps all-ids)]
                         :when (seq missing)]
                     {:dev-id dev-id :missing-deps missing})]
    (when (seq violations)
      {:invariant :deps-exist
       :violation :missing-dependencies
       :details violations
       :severity :error
       :message (str "Dependencies reference non-existent entities: "
                     (set (mapcat :missing-deps violations)))})))

(defn invariant-no-circular-deps
  "Dependency graph must be acyclic."
  []
  (let [all-ids (all-dev-ids)
        deps-map (into {} (map (fn [id] [id (rt/deps-for id)]) all-ids))]
    (letfn [(has-cycle? [id visited path]
              (cond
                (contains? path id) {:cycle (conj (vec path) id)}
                (contains? visited id) nil
                :else (some #(has-cycle? % (conj visited id) (conj path id))
                            (get deps-map id #{}))))]
      (when-let [cycle (some #(has-cycle? % #{} []) all-ids)]
        {:invariant :no-circular-deps
         :violation :dependency-cycle
         :cycle (:cycle cycle)
         :severity :error
         :message (str "Dependency cycle detected: " (:cycle cycle))}))))

;; =============================================================================
;; REACHABILITY AXIOMS
;; =============================================================================

(defn invariant-all-fns-reachable
  "Every function should be reachable from some endpoint.
   
   Excludes functions marked with :lifecycle/* aspects (setup, migration, etc.)"
  []
  (let [endpoints (rt/all-with-aspect :semantic-namespace/endpoint)
        all-fns (set (rt/all-with-aspect :semantic-namespace/function))
        ;; Exclude lifecycle functions (not meant to be endpoint-reachable)
        lifecycle-fns (->> @cid/registry
                           (filter (fn [[id _]]
                                     (some #(= "lifecycle" (namespace %)) id)))
                           (map (fn [[_ v]] (:dev/id v)))
                           set)
        checkable-fns (set/difference all-fns lifecycle-fns)
        ;; Find reachable via BFS from endpoints
        reachable (atom #{})
        collect-reachable (fn collect [id]
                            (when-not (@reachable id)
                              (swap! reachable conj id)
                              (doseq [dep (rt/deps-for id)]
                                (collect dep))))]
    (doseq [ep endpoints]
      (collect-reachable ep))
    (let [unreachable (set/difference checkable-fns @reachable)]
      (when (seq unreachable)
        {:invariant :all-fns-reachable
         :violation :unreachable-functions
         :functions unreachable
         :severity :warning
         :message (str "Functions not reachable from any endpoint: " unreachable
                       " (mark with :lifecycle/* if intentional)")}))))

;; =============================================================================
;; TIER AXIOMS (unchanged)
;; =============================================================================

(defn invariant-components-are-foundation
  "Components should be :tier/foundation."
  []
  (let [components (rt/all-with-aspect :semantic-namespace/component)
        violations (remove #(rt/has-aspect? % :tier/foundation) components)]
    (when (seq violations)
      {:invariant :components-are-foundation
       :violation :wrong-tier
       :components violations
       :severity :error
       :message (str "Components should be :tier/foundation: " violations)})))

(defn invariant-endpoints-are-api-tier
  "Endpoints should be :tier/api."
  []
  (let [endpoints (rt/all-with-aspect :semantic-namespace/endpoint)
        violations (remove #(rt/has-aspect? % :tier/api) endpoints)]
    (when (seq violations)
      {:invariant :endpoints-are-api-tier
       :violation :wrong-tier
       :endpoints violations
       :severity :error
       :message (str "Endpoints should be :tier/api: " violations)})))

;; =============================================================================
;; SEMANTIC CONSISTENCY AXIOMS
;; =============================================================================

(defn invariant-external-is-async
  "Functions marked :integration/external should also be :temporal/async."
  []
  (let [external-fns (filter #(rt/has-aspect? % :semantic-namespace/function)
                             (rt/all-with-aspect :integration/external))
        violations (remove #(rt/has-aspect? % :temporal/async) external-fns)]
    (when (seq violations)
      {:invariant :external-is-async
       :violation :external-not-async
       :functions violations
       :severity :warning
       :message (str "External integrations should be async: " violations)})))

(defn invariant-pure-has-no-deps
  "Functions marked :effect/pure should have no component dependencies."
  []
  (let [pure-fns (filter #(rt/has-aspect? % :effect/pure)
                         (rt/all-with-aspect :semantic-namespace/function))
        violations (for [fn-id pure-fns
                         :let [deps (rt/deps-for fn-id)
                               component-deps (filter #(rt/has-aspect? % :semantic-namespace/component) deps)]
                         :when (seq component-deps)]
                     {:fn fn-id :component-deps (vec component-deps)})]
    (when (seq violations)
      {:invariant :pure-has-no-deps
       :violation :pure-fn-with-deps
       :details violations
       :severity :error
       :message "Pure functions should not depend on components"})))

;; =============================================================================
;; PROTOCOL AXIOMS
;; =============================================================================

(defn get-protocol-aspects
  "Extract protocol aspects from a compound identity.
   Returns seq of keywords with namespace 'protocol'."
  [compound-id]
  (filter #(= "protocol" (namespace %)) compound-id))

(defn invariant-protocol-exists
  "Components that declare protocol aspects must have those protocols registered."
  []
  (let [components-with-protocols (->> @cid/registry
                                       (filter (fn [[id _]]
                                                 (and (contains? id :semantic-namespace/component)
                                                      (seq (get-protocol-aspects id)))))
                                       (map (fn [[id v]]
                                              {:dev-id (:dev/id v)
                                               :protocols (vec (get-protocol-aspects id))})))
        all-protocol-ids (->> @cid/registry
                             (filter (fn [[id _]] (contains? id :semantic-namespace/protocol)))
                             (map (fn [[_ v]] (:dev/id v)))
                             set)
        violations (for [{:keys [dev-id protocols]} components-with-protocols
                         protocol protocols
                         :when (not (contains? all-protocol-ids protocol))]
                     {:component dev-id :missing-protocol protocol})]
    (when (seq violations)
      {:invariant :protocol-exists
       :violation :missing-protocol-definition
       :details violations
       :severity :error
       :message (str "Components reference undefined protocols: "
                     (set (map :missing-protocol violations)))})))

(defn invariant-protocol-conformance
  "Components implementing a protocol must provide all required protocol functions.

   Checks that component's implementation map contains all methods declared
   in the protocol's :protocol/functions."
  []
  (let [protocols (->> @cid/registry
                      (filter (fn [[id _]] (contains? id :semantic-namespace/protocol)))
                      (map (fn [[_ v]]
                             {:protocol-id (:dev/id v)
                              :required-fns (set (:protocol/functions v))}))
                      (remove #(empty? (:required-fns %))))

        ;; Note: This invariant is for design-time checking. At runtime, implementations
        ;; are typically functions/closures, not maps with protocol methods.
        ;; This invariant validates that component REGISTRATIONS declare conformance.
        violations (for [{:keys [protocol-id required-fns]} protocols
                         :when required-fns
                         [compound-id value] @cid/registry
                         :when (and (contains? compound-id :semantic-namespace/component)
                                   (contains? compound-id protocol-id))
                         :let [dev-id (:dev/id value)
                               ;; Components may provide protocol methods in their value map
                               ;; or may be validated elsewhere (implementation-specific)
                               provided-methods (when (map? value)
                                                 (set (filter #(namespace %) (keys value))))
                               missing (when provided-methods
                                        (set/difference required-fns provided-methods))]
                         :when (and missing (seq missing))]
                     {:component dev-id
                      :protocol protocol-id
                      :missing-methods (vec missing)})]
    (when (seq violations)
      {:invariant :protocol-conformance
       :violation :incomplete-protocol-implementation
       :details violations
       :severity :warning
       :message (str "Components don't implement all protocol methods. "
                     "This is a warning because implementations may be provided at runtime.")})))

;; =============================================================================
;; CHECK ALL
;; =============================================================================

(def all-invariants
  "All invariants in check order."
  [;; Structural
   invariant-deps-exist
   invariant-no-circular-deps
   ;; Tier
   invariant-components-are-foundation
   invariant-endpoints-are-api-tier
   ;; Dataflow
   invariant-context-satisfiable
   invariant-no-orphan-responses        ; ← uses fixed version
   ;; Reachability
   invariant-all-fns-reachable
   ;; Semantic consistency
   invariant-external-is-async
   invariant-pure-has-no-deps
   ;; Protocol conformance
   invariant-protocol-exists
   invariant-protocol-conformance])

(defn check-all
  "Run all invariants, return {:valid? :errors :warnings :violations}."
  []
  (let [results (keep #(%) all-invariants)
        errors (filter #(= :error (:severity %)) results)
        warnings (filter #(= :warning (:severity %)) results)]
    {:violations results
     :errors errors
     :warnings warnings
     :valid? (empty? errors)}))

(defn report
  "Print human-readable invariant report."
  []
  (let [{:keys [errors warnings valid?]} (check-all)]
    (println "\n=== AXIOM VALIDATION REPORT ===\n")
    (if valid?
      (println "✓ All error-level invariants pass")
      (do
        (println "✗ ERRORS:")
        (doseq [e errors]
          (println "  -" (:invariant e) ":" (:message e)))))
    (when (seq warnings)
      (println "\n⚠ WARNINGS:")
      (doseq [w warnings]
        (println "  -" (:invariant w) ":" (:message w))))
    (println "\nTotal:" (+ (count errors) (count warnings))
             "issues (" (count errors) "errors," (count warnings) "warnings)")
    valid?))
