(ns atlas.graph
  "Runtime that drives behavior from semantic registry."
  (:require [atlas.registry :as registry]
            [atlas.query :as query]))

(defn fetch-by-dev-id
  "Find the [identity value] pair for a given dev-id.
   Delegates to atlas.query/find-by-dev-id."
  [dev-id]
  (query/find-by-dev-id @registry/registry dev-id))

(defn identity-for [dev-id]
  (first (fetch-by-dev-id dev-id)))

(defn props-for [dev-id]
  (second (fetch-by-dev-id dev-id)))

(defn has-aspect? [dev-id aspect]
  (contains? (identity-for dev-id) aspect))

(defn all-with-aspect
  "Find all dev-ids with given aspect.
   Delegates to atlas.query/find-dev-ids-with-aspect."
  [aspect]
  (query/find-dev-ids-with-aspect @registry/registry aspect))

(defn deps-for [dev-id]
  (or (:semantic-namespace/deps (props-for dev-id)) #{}))

(defn topo-sort
  "Topological sort of dev-ids by dependencies."
  [dev-ids]
  (let [deps-map (into {} (map (fn [id] [id (deps-for id)]) dev-ids))
        sorted (atom [])
        visited (atom #{})]
    (letfn [(visit [id]
              (when-not (@visited id)
                (swap! visited conj id)
                (doseq [dep (get deps-map id #{})]
                  (visit dep))
                (swap! sorted conj id)))]
      (doseq [id dev-ids]
        (visit id)))
    @sorted))

(defn trace-data-flow
  "Trace what produces each context key for a function.
   Uses atlas.query for finding producers."
  [dev-id]
  (let [props (props-for dev-id)
        context (or (:semantic-namespace/context props) [])]
    (for [ctx-key context
          :let [producers-map (query/find-producers @registry/registry ctx-key :semantic-namespace/response)
                function-producers (query/find-by-aspect @registry/registry :semantic-namespace/function)
                producer-ids (->> producers-map
                                  (keep (fn [[id v]]
                                          (when (contains? function-producers id)
                                            (:dev/id v))))
                                  vec)]]
      {:needs ctx-key
       :produced-by producer-ids
       :satisfied? (seq producer-ids)})))

(defn compute-data-deps
  "Compute dependencies based on context/response data flow.
   Uses atlas.query for finding data flow connections."
  [dev-id]
  (let [props (props-for dev-id)
        context (set (:semantic-namespace/context props))
        all-fns (all-with-aspect :semantic-namespace/function)]
    (->> context
         (mapcat (fn [ctx-key]
                   (let [producers (query/find-producers @registry/registry ctx-key :semantic-namespace/response)]
                     (->> producers
                          (keep (fn [[_ v]] (:dev/id v)))
                          (filter all-fns)))))
         set)))

(defn topo-sort-by-data
  "Topological sort based on data flow (context/response)."
  [dev-ids]
  (let [dev-ids-set (set dev-ids)
        deps-map (into {} (map (fn [id] [id (compute-data-deps id)]) dev-ids))
        sorted (atom [])
        visited (atom #{})]
    (letfn [(visit [id]
              (when-not (@visited id)
                (swap! visited conj id)
                (doseq [dep (get deps-map id #{})]
                  (when (contains? dev-ids-set dep)
                    (visit dep)))
                (swap! sorted conj id)))]
      (doseq [id dev-ids]
        (visit id)))
    @sorted))
