package org.bukkit.command.defaults;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.command.CommandSender;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;
// Spigot start
import net.kyori.adventure.text.Component;
import net.kyori.adventure.text.TextComponent;
import net.kyori.adventure.text.format.NamedTextColor;
import net.kyori.adventure.text.event.ClickEvent;
import org.bukkit.entity.Player;
import org.bukkit.plugin.PluginDescriptionFile;
// Spigot end

public class PluginsCommand extends BukkitCommand {
    public PluginsCommand(@NotNull String name) {
        super(name);
        this.description = "Gets a list of plugins running on the server";
        this.usageMessage = "/plugins";
        this.setPermission("bukkit.command.plugins");
        this.setAliases(Arrays.asList("pl"));
    }

    @Override
    public boolean execute(@NotNull CommandSender sender, @NotNull String currentAlias, @NotNull String[] args) {
        if (!testPermission(sender)) return true;

        // Spigot start
        if (sender instanceof Player && sender.hasPermission("bukkit.command.version")) {
            sender.sendMessage(getPluginListSpigot());
        } else {
            sender.sendMessage("Plugins " + getPluginList());
        }
        // Spigot end
        return true;
    }

    @NotNull
    @Override
    public List<String> tabComplete(@NotNull CommandSender sender, @NotNull String alias, @NotNull String[] args) throws IllegalArgumentException {
        return Collections.emptyList();
    }

    @NotNull
    private String getPluginList() {
        // Paper start
        TreeMap<String, Plugin> plugins = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);

        for (Plugin plugin : Bukkit.getPluginManager().getPlugins()) {
            plugins.put(plugin.getDescription().getName(), plugin);
        }

        StringBuilder pluginList = new StringBuilder();
        for (Map.Entry<String, Plugin> entry : plugins.entrySet()) {
            if (pluginList.length() > 0) {
                pluginList.append(ChatColor.WHITE);
                pluginList.append(", ");
            }

            Plugin plugin = entry.getValue();

            pluginList.append(plugin.isEnabled() ? ChatColor.GREEN : ChatColor.RED);
            // Paper start - Add an asterisk to legacy plugins (so admins are aware)
            String pluginName = plugin.getDescription().getName();
            if (org.bukkit.UnsafeValues.isLegacyPlugin(plugin)) {
                pluginName += "*";
            }
            pluginList.append(pluginName);
            // Paper end

            if (plugin.getDescription().getProvides().size() > 0) {
                pluginList.append(" (").append(String.join(", ", plugin.getDescription().getProvides())).append(")");
            }
        }

        return "(" + plugins.size() + "): " + pluginList.toString();
        // Paper end
    }

    // Spigot start
    @NotNull
    private TextComponent getPluginListSpigot() {
        Plugin[] plugins = Arrays.stream(Bukkit.getPluginManager().getPlugins()).sorted(java.util.Comparator.comparing(plugin -> plugin.getName().toLowerCase())).toArray(Plugin[]::new); // Purpur
        TextComponent.Builder builder = Component.text();
        builder.append(Component.text("Plugins (" + plugins.length + "): "));

        int index = 0;
        for (Plugin plugin : plugins) {
            if (index++ > 0) {
                builder.append(Component.text(", ", NamedTextColor.WHITE));
            }

            // Event components
            PluginDescriptionFile description = plugin.getDescription();
            TextComponent.Builder hover = Component.text();
            hover.append(Component.text("Version: ", NamedTextColor.WHITE)).append(Component.text(description.getVersion(), NamedTextColor.GREEN));

            if (description.getDescription() != null) {
                hover.append(Component.newline())
                    .append(Component.text("Description: ", NamedTextColor.WHITE))
                    .append(Component.text(description.getDescription(), NamedTextColor.GREEN));
            }

            if (description.getWebsite() != null) {
                hover.append(Component.newline())
                    .append(Component.text("Website: ", NamedTextColor.WHITE))
                    .append(Component.text(description.getWebsite(), NamedTextColor.GREEN));
            }

            if (!description.getAuthors().isEmpty()) {
                hover.append(Component.newline());
                if (description.getAuthors().size() == 1) {
                    hover.append(Component.text("Author: "));
                } else {
                    hover.append(Component.text("Authors: "));
                }

                hover.append(getAuthors(description));
            }

            // Plugin list entry
            builder.append(Component.text(plugin.getDescription().getName(), plugin.isEnabled() ? NamedTextColor.GREEN : NamedTextColor.RED)
                .hoverEvent(hover.build()).clickEvent(ClickEvent.suggestCommand("/version " + description.getName())));

            if (plugin.getDescription().getProvides().size() > 0) {
                builder.append(Component.text(" (", NamedTextColor.WHITE)).append(Component.text(String.join(", ", plugin.getDescription().getProvides()))).append(Component.text(")"));
            }
        }

        return builder.build();
    }

    @NotNull
    private TextComponent getAuthors(@NotNull final PluginDescriptionFile description) {
        TextComponent.Builder builder = Component.text();
        List<String> authors = description.getAuthors();

        for (int i = 0; i < authors.size(); i++) {
            if (i > 0) {
                builder.append(Component.text(i < authors.size() - 1 ? ", " : " and ", NamedTextColor.WHITE));
            }

            builder.append(Component.text(authors.get(i), NamedTextColor.GREEN));
        }

        return builder.build();
    }
    // Spigot end
}
