(ns f-form.validation.vlad
  "Tools for validating forms and their fields with
  https://github.com/logaan/vlad.

  If you wish to use this namespace, you must provide `vlad` in your own
  dependencies. If you want to validate with another library, do not load this
  namespace."
  (:require [clojure.string :as string]
            [f-form.form :as form]
            [f-form.validation :as validation]
            [vlad.core :as vlad]))

(defn assign-names
  "_Helper:_ augments `errors` with field labels. These labels will be used to
  generate the error messages.

  The `field-labels` work in coordination with [[f-form.validation.vlad/field]].
  Suppose you have the following:

  ``` clojure
  (def form
    (form/init [(field/init [:address/street])]))

  (def validation
    (f-form.validation.vlad/field [:address/street]
                                  (f-form.validation.vlad/non-nil)))

  (def field-labels
    {[:address/street] \"Street\"})
  ```

  Then:

  ``` clojure
  (f-form.validation.vlad/validate form validation field-labels)
  ;; => {:form/fields {:address/street {:field/errors [\"Street is required.\"] ,,,}
  ;;     :form/fields-valid? false}
  ```

  This function has special handling for complex fields. Suppose you are
  collecting a birthday, like so:

  ``` clojure
  (def form
    (form/init [(field/init [:person/birthday]
                            {:date/month nil
                             :date/day   nil})]))

  (def validation
    (f-form.validation.vlad/field [:person/birthday]
                                  (vlad/chain
                                    (vlad/attr [:date/month] (vlad/present))
                                    (vlad/attr [:date/year] (vlad/present)))))
  ```

  The first option is to label the field as a whole:

  ``` clojure
  (def field-labels
    {[:person/birthday] \"Birthday\"})
  ```

  Then validation will be translated like so:

  ``` clojure
  (f-form.validation.vlad/validate form validation field-labels)
  ;; => {:form/fields {:person/birthday {:field/errors [\"Birthday is required.\"] ,,,}
  ;;     :form/fields-valid? false}
  ```

  Alternatively, if the individual components of a complex-valued field need
  their own labels, they can be provided like so:

  ```clojure

  (def field-labels
    {[:person/birthday] {[:date/month] \"Birth month\"
                         [:date/year] \"Birth year\"}})

  (f-form.validation.vlad/validate form validation field-labels)
  ;; => {:form/fields {:person/birthday {:field/errors [\"Birth month is required.\"]}
  ;;     :form/fields-valid? false}
  ```
  "
  [errors field-labels]
  (map (fn [e]
         (if (:name e)
           e
           (assoc e :name (let [name-or-submap (field-labels (:selector e))]
                            (if (map? name-or-submap)
                              (name-or-submap (:value-selector e))
                              name-or-submap)))))
       errors))

(defrecord Value [validation]
  vlad/Validation
  (validate [_ data]
    (map (fn [error]
           ;; Ensure `:selector` is the `:field/path`, and `:value-selector`
           ;; is the path within the `:field/value`. These two selectors are
           ;; used by [[assign-names]] to customize the names of invalid fields.
           (-> error
               (assoc :value-selector (:selector error))
               (dissoc :selector)))
         (vlad/validate validation (get data :field/value)))))

(defn value
  "_Validation wrapper:_ Runs a `validation` on the `:field/value` of the field
  specified in the surrounding attr. In most cases you should prefer [[field]]."
  ([] (Value. vlad/valid))
  ([validation] (Value. validation)))

(defn field
  "_Validation wrapper:_ Runs a `validation` on the `:field/value` of a field
  whose `:field/path` matches the given `field-path`."
  ([field-path]            (vlad/attr field-path (value)))
  ([field-path validation] (vlad/attr field-path (value validation))))

(defrecord Urgency [level validation]
  vlad/Validation
  (validate [_ data]
    (map (fn [error]
           (assoc error :urgency level))
         (vlad/validate validation data))))

(defn urgency
  "_Validation wrapper:_ Sets the urgency for any errors generated by `validation`
  to `level`."
  [level validation]
  (Urgency. level validation))

(defn warning
  "_Validation wrapper:_ Sets the urgency for any errors generated by `validation`
  to `::warning`. This controls whether [[validate]] puts the errors on
  `:field/errors` or `:field/warnings` and influences the translation of the
  error."
  [validation]
  (urgency ::warning validation))

(defn not-pristine
  "_Validator:_ Checks that the field is not pristine. Should be used on a field,
  not a field's value, so:

  ```clojure
  ;; do this
  (vlad/attr [:postal-code] (form.validation/not-pristine))
  ;; not this
  (form.validation/field [:postal-code] (form.validation/not-pristine))
  ```
  "
  ([]
   (not-pristine {}))
  ([error-data]
   (vlad/predicate :field/pristine?
                   (merge {:type ::not-pristine} error-data))))

(defn value-in
  "_Validator:_ Checks that the value is over `low` and under `high`, inclusive of both. No
  checking is done that `low` is lower than `high`."
  ([low high]
   (value-in low high {}))
  ([low high error-data]
   (vlad/predicate #(or (< high %)
                        (> low %))
                   (merge {:type ::value-in :low low :high high} error-data))))

(defn non-nil
  "_Validator:_ Checks that the value is not nil."
  ([]
   (non-nil {}))
  ([error-data]
   (vlad/predicate nil?
                   (merge {:type ::vlad/present} error-data))))

(defn is-uuid
  "_Validator:_ Checks that the value is a uuid."
  ([]
   (is-uuid {}))
  ([error-data]
   (vlad/predicate (complement uuid?)
                   (merge {:type ::is-uuid} error-data))))

(defn is-inst
  "_Validator:_ Checks that the value is an instant."
  ([]
   (is-inst {}))
  ([error-data]
   (vlad/predicate (complement inst?)
                   (merge {:type ::vlad/present} error-data))))

(defn- pos-number? [x]
  (and (number? x)
       (> x 0)))

(defn pos-number
  "_Validator:_ Checks that the value is a positive number."
  ([]
   (pos-number {}))
  ([error-data]
   (vlad/predicate (complement pos-number?)
                   (merge {:type ::pos-number} error-data))))

(defmulti english-urgency-force
  "_Helper:_ Adjusts the strength of the language used in warnings."
  {:no-doc   true
   :arglists '([error])}
  :urgency)
(defmethod english-urgency-force :default [_] "must")
(defmethod english-urgency-force ::warning [_] "should")

(defmulti english-urgency-necessity
  "_Helper:_ Adjusts the strength of the language used in warnings."
  {:no-doc   true
   :arglists '([error])}
  :urgency)
(defmethod english-urgency-necessity :default [_] "required")
(defmethod english-urgency-necessity ::warning [_] "expected")

(defmulti english-translation
  "_Helper:_ Takes an error and returns a human readable version of it.

  Adapted from
  [vlad.core/english-translation](https://github.com/logaan/vlad/blob/1dd8d427578655148e5333b3e40f0fe0223ee5eb/src/vlad/core.cljc#L224)
  to account for f-form specific errors and urgency levels."
  {:arglists '([error])}
  :type)

(defmethod english-translation :vlad.core/present
  [{:keys [name] :as error}]
  (str name " is " (english-urgency-necessity error) "."))

(defmethod english-translation :vlad.core/length-over
  [{:keys [name size] :as error}]
  (str name " " (english-urgency-force error) " be over " size " characters long."))

(defmethod english-translation :vlad.core/length-under
  [{:keys [name size] :as error}]
  (str name " " (english-urgency-force error) " be under " size " characters long."))

(defmethod english-translation :vlad.core/one-of
  [{:keys [name set] :as error}]
  (str name " " (english-urgency-force error) " be one of " (string/join ", " set) "."))

(defmethod english-translation :vlad.core/not-of
  [{:keys [name set] :as error}]
  (str name " " (english-urgency-force error) " not be one of " (string/join ", " set) "."))

(defmethod english-translation :vlad.core/equals-value
  [{:keys [name value] :as error}]
  (str name " " (english-urgency-force error) " be \"" value "\"."))

(defmethod english-translation :vlad.core/equals-field
  [{:keys [first-name second-name] :as error}]
  (str first-name " " (english-urgency-force error) " be the same as " second-name "."))

(defmethod english-translation :vlad.core/matches
  [{:keys [name pattern] :as error}]
  (str name " " (english-urgency-force error) " match the pattern " (.toString pattern) "."))

(defmethod english-translation ::not-pristine [{:keys [name] :as error}]
  (str name " " (english-urgency-force error) " be changed."))

(defmethod english-translation ::value-in [{:keys [name low high] :as error}]
  (str name " " (english-urgency-force error) " be between " low " and " high "."))

(defmethod english-translation ::is-uuid [{:keys [name] :as error}]
  (str name " " (english-urgency-force error) " be an id."))

(defmethod english-translation ::pos-number [{:keys [name] :as error}]
  (str name " " (english-urgency-force error) " be a positive number."))

(defn validate
  "_Core:_ Applies a `validation` to a `form`. Adds `:field/errors` and
  `:field/warnings` to invalid fields. Updates the whole form's
  `:form/fields-valid?` depending on whether there are any `:field/errors`.

  Similar to
  [vlad.core/field-errors](https://github.com/logaan/vlad/blob/1dd8d427578655148e5333b3e40f0fe0223ee5eb/src/vlad/core.cljc#L282),
  but customized for dealing with forms.

  Errors and warnings are vectors of English strings, translated using the
  `field-labels` (according to [[assign-names]]) and
  [vlad.core/translate-errors](https://github.com/logaan/vlad/blob/1dd8d427578655148e5333b3e40f0fe0223ee5eb/src/vlad/core.cljc#L260)
  using the `translation` (by default [[english-translation]]).

  See
  [vlad.core/translate-errors](https://github.com/logaan/vlad/blob/1dd8d427578655148e5333b3e40f0fe0223ee5eb/src/vlad/core.cljc#L260)
  for additional detail about error message generation."
  ([form validation field-labels]
   (validate form validation field-labels english-translation))
  ([form validation field-labels translation]
   (let [{errors   ::error
          warnings ::warning} (->> (vlad/validate validation (:form/fields form))
                                   (group-by #(get % :urgency ::error)))

         errors   (-> errors
                      (assign-names field-labels)
                      (vlad/translate-errors translation))
         warnings (-> warnings
                      (assign-names field-labels)
                      (vlad/translate-errors translation))]
     (reduce (fn [form path]
               (form/update-field-by-path form path
                                          (fn [field]
                                            (-> field
                                                (assoc :field/errors (get errors path))
                                                (assoc :field/warnings (get warnings path))))))
             (validation/set-valid form (empty? errors))
             (:form/field-paths form)))))
