(ns telsos.lib.es.store.memory
  (:require
   [telsos.lib.es.store :as store]
   [telsos.lib.uuid :as uuid]
   [tick.core :as t]))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

(defn- now []
  (t/now))

(defn- find-event-conflict
  "Returns true if an event with the same agg :type, :id, :ver exists."
  [events agg-type agg-id agg-ver]
  (some (fn [e]
          (let [agg (:agg e)]
            (and (= (:type agg) agg-type)
                 (= (:id   agg) agg-id)
                 (= (:ver  agg) agg-ver))))
        events))

(defn- insert-event-cas!
  "Atomically inserts event using compare-and-set loop.
   Throws on version conflict. Returns result map on success."
  [state event result]
  (let [{agg-type :type agg-id :id agg-ver :ver} (:agg event)]
    (loop []
      (let [current @state]
        (if (find-event-conflict (:events current) agg-type agg-id agg-ver)
          (throw (ex-info "Event version conflict"
                          {:agg-type agg-type
                           :agg-id   agg-id
                           :agg-ver  agg-ver}))
          (if (compare-and-set! state current (update current :events conj event))
            result
            (recur)))))))

(defn- update-agg-cas!
  "Atomically updates aggregate using compare-and-set loop.
   Returns {:valid-from ...} on success, nil if not found."
  [state agg-type id ver attrs valid-from]
  (loop []
    (let [current @state]
      (when-let [existing (get-in current [:aggs agg-type id])]
        (let [updated
              (assoc existing
                     :ver        ver
                     :attrs      attrs
                     :valid-from valid-from)

              new-state (assoc-in current [:aggs agg-type id] updated)]

          (if (compare-and-set! state current new-state)
            {:valid-from valid-from}
            (recur)))))))

(defrecord MemoryStore [state]
  store/Store

  (insert-event! [_ params]
    (let [{:keys [type request-id interaction-id payload]
           {agg-type :type agg-id :id agg-ver :ver} :agg}
          params

          id         (uuid/uuidv7)
          created-at (now)
          event      {:id             id
                      :type           type
                      :agg            {:id   agg-id
                                       :type agg-type
                                       :ver  agg-ver}
                      :request-id     request-id
                      :interaction-id interaction-id
                      :created-at     created-at
                      :payload        payload}]

      (insert-event-cas! state event {:id id :created-at created-at})))

  (select-events-by-agg-id [_ agg-type agg-id]
    (->> (:events @state)
         (filter (fn [e]
                   (and (= (-> e :agg :type) agg-type)
                        (= (-> e :agg   :id) agg-id))))
         (sort-by (comp :ver :agg))))

  (select-event-by-agg-version [_ agg-type agg-id agg-ver]
    (->> (:events @state)
         (filter (fn [e]
                   (and (= (-> e :agg :type) agg-type)
                        (= (-> e :agg   :id) agg-id)
                        (= (-> e :agg  :ver) agg-ver))))
         first))

  (insert-agg! [_ agg-type attrs]
    (let [id         (uuid/uuidv7)
          valid-from (now)
          agg        {:id         id
                      :type       agg-type
                      :ver        1
                      :attrs      attrs
                      :valid-from valid-from}]
      (swap! state assoc-in [:aggs agg-type id] agg)
      {:id id :valid-from valid-from}))

  (update-agg! [_ agg-type id ver attrs]
    (update-agg-cas! state agg-type id ver attrs (now)))

  (select-agg-by-id [_ agg-type id]
    (get-in @state [:aggs agg-type id])))

(defn create-memory-store
  "Create a new in-memory store for testing."
  []
  (->MemoryStore (atom {:events [] :aggs {}})))
