(ns telsos.lib.es
  #?(:cljs
     (:require-macros
      [telsos.lib.assertions :refer [asrt maybe the]]))
  (:require
   [malli.core :as m]
   [telsos.lib.edn-json :as edn-json :refer [JsonEdn]]
   [telsos.lib.es.aggs :as aggs]
   [telsos.lib.es.events :as events]
   [telsos.lib.es.store :as store]
   #?(:clj [telsos.lib.assertions :refer [asrt maybe the]])))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

;; REFS
(def Ref
  [:map {:closed true}
   [:id   :uuid]
   [:type :keyword]
   [:ver  {:optional true} [:int {:min 1}]]])

(def VerRef
  [:map {:closed true}
   [:id   :uuid]
   [:type :keyword]
   [:ver  [:int {:min 1}]]])

;; EVENTS
(def Event
  [:map {:closed true}
   [:id             :uuid]
   [:type        :keyword]
   [:agg           VerRef]
   [:request-id     :uuid]
   [:interaction-id :uuid]
   [:created-at     :inst]
   [:payload      JsonEdn]])

(def Event? (m/validator Event {:registry edn-json/json-edn-registry}))

(defn add-event!
  "Insert a new event. Returns {:id uuid :created-at timestamp}.
   Throws on (agg :type, agg :id, agg :ver) conflict."
  [store {:keys [type request-id interaction-id payload]
          {agg-type :type agg-id :id agg-ver :ver} :agg
          :as params}]

  (the events/event-type? type)
  (the aggs/agg-type? agg-type)
  (the uuid?            agg-id)
  (the pos-int?        agg-ver)
  (the uuid?        request-id)
  (the uuid?    interaction-id)

  (asrt payload)
  (edn-json/the-JsonEdn payload)

  (store/insert-event! store params))

(defn get-events-by-agg-id
  [store agg-type agg-id]
  (the aggs/agg-type? agg-type)
  (the uuid?            agg-id)

  (map #(the Event? %)
       (store/select-events-by-agg-id store agg-type agg-id)))

(defn get-event-by-agg-version
  [store agg-type agg-id agg-ver]
  (the aggs/agg-type? agg-type)
  (the uuid?            agg-id)
  (the pos-int?       agg-ver)

  (maybe Event?
         (store/select-event-by-agg-version store agg-type agg-id agg-ver)))

;; AGGREGATES
(def Agg
  [:map {:closed true}
   [:id             :uuid]
   [:type        :keyword]
   [:ver   [:int {:min 1}]]
   [:attrs        JsonEdn]
   [:valid-from     :inst]])

(def Agg? (m/validator Agg {:registry edn-json/json-edn-registry}))

(defn add-agg!
  "Insert a new aggregate. Returns {:id uuid :valid-from timestamp}."
  [store agg-type agg-attrs]
  (the aggs/agg-type?    agg-type)
  (the map?             agg-attrs)
  (edn-json/the-JsonEdn agg-attrs)

  (store/insert-agg! store agg-type agg-attrs))

(defn modify-agg!
  "Update an existing aggregate. Returns {:valid-from timestamp} or nil if not found."
  [store agg-type agg-id agg-ver agg-attrs]
  (the aggs/agg-type?    agg-type)
  (the uuid?               agg-id)
  (the pos-int?           agg-ver)
  (the map?             agg-attrs)
  (edn-json/the-JsonEdn agg-attrs)

  (store/update-agg! store agg-type agg-id agg-ver agg-attrs))

(defn get-agg-by-id
  "Select aggregate by id. Returns {:id :type :ver :attrs :valid-from} or nil."
  [store agg-type agg-id]
  (the aggs/agg-type? agg-type)
  (the uuid?            agg-id)
  (maybe Agg?
         (store/select-agg-by-id store agg-type agg-id)))
