(ns telsos.lib.es.store.memory
  (:require
   [telsos.lib.es.store :as store]
   [telsos.lib.uuid :as uuid]
   [tick.core :as t]))

#?(:clj (set! *warn-on-reflection*       true))
#?(:clj (set! *unchecked-math* :warn-on-boxed))

(defn- str->uuid [s]
  #?(:clj  (java.util.UUID/fromString s)
     :cljs (cljs.core/uuid s)))

(defn- now []
  (t/inst (t/now)))

(defn- find-event-conflict
  "Returns true if an event with the same agg-type, agg-id, agg-version exists."
  [events agg-type agg-id agg-version]
  (some (fn [e]
          (and (= (:agg-type e) agg-type)
               (= (:agg-id e) agg-id)
               (= (:agg-version e) agg-version)))
        events))

(defrecord MemoryStore [state]
  store/Store

  (insert-event! [_ params]
    (let [{:keys [event-type agg-type agg-id agg-version
                  request-id interaction-id payload]} params
          id         (str->uuid (uuid/uuidv7))
          created-at (now)
          event      {:id             id
                      :event-type     event-type
                      :agg-type       agg-type
                      :agg-id         agg-id
                      :agg-version    agg-version
                      :request-id     request-id
                      :interaction-id interaction-id
                      :created-at     created-at
                      :payload        payload}]
      (swap! state
             (fn [s]
               (when (find-event-conflict (:events s) agg-type agg-id agg-version)
                 (throw (ex-info "Event version conflict"
                                 {:agg-type    agg-type
                                  :agg-id      agg-id
                                  :agg-version agg-version})))
               (update s :events conj event)))
      {:id id :created-at created-at}))

  (select-events-by-agg-id [_ agg-type agg-id]
    (->> (:events @state)
         (filter (fn [e]
                   (and (= (:agg-type e) agg-type)
                        (= (:agg-id e) agg-id))))
         (sort-by :agg-version)))

  (select-event-by-agg-version [_ agg-type agg-id agg-version]
    (->> (:events @state)
         (filter (fn [e]
                   (and (= (:agg-type e) agg-type)
                        (= (:agg-id e) agg-id)
                        (= (:agg-version e) agg-version))))
         first))

  (insert-agg! [_ agg-type attrs]
    (let [id         (str->uuid (uuid/uuidv7))
          valid-from (now)
          agg        {:id         id
                      :agg-type   agg-type
                      :version    1
                      :attrs      attrs
                      :valid-from valid-from}]
      (swap! state assoc-in [:aggs agg-type id] agg)
      {:id id :valid-from valid-from}))

  (update-agg! [_ agg-type id version attrs]
    (let [valid-from (now)
          result     (atom nil)]
      (swap! state
             (fn [s]
               (if-let [existing (get-in s [:aggs agg-type id])]
                 (let [updated (assoc existing
                                      :version    version
                                      :attrs      attrs
                                      :valid-from valid-from)]
                   (reset! result {:valid-from valid-from})
                   (assoc-in s [:aggs agg-type id] updated))
                 s)))
      @result))

  (select-agg-by-id [_ agg-type id]
    (get-in @state [:aggs agg-type id])))

(defn create-memory-store
  "Create a new in-memory store for testing."
  []
  (->MemoryStore (atom {:events [] :aggs {}})))
